package main

import (
	"fmt"
	"net"
	"strconv"
	"strings"

	"code.google.com/p/gopacket/layers"
)

type Uint16Range struct {
	bottom uint16
	top    uint16
}

func (r *Uint16Range) Contains(n uint16) bool {
	return n >= r.bottom && n <= r.top
}

func ParseUint16Range(s string) (*Uint16Range, error) {
	bookends := strings.Split(s, "-")
	r := &Uint16Range{}
	if len(bookends) > 2 {
		return nil, fmt.Errorf("invalid numeric range %s", s)
	} else if len(bookends) == 1 {
		n, err := strconv.ParseUint(bookends[0], 10, 16)
		if err != nil {
			return nil, err
		}
		r.top = uint16(n)
		r.bottom = uint16(n)
	} else {
		n, err := strconv.ParseUint(bookends[0], 10, 16)
		if err != nil {
			return nil, err
		}
		r.bottom = uint16(n)
		n, err = strconv.ParseUint(bookends[1], 10, 16)
		if err != nil {
			return nil, err
		}
		r.top = uint16(n)
	}
	return r, nil
}

type Filter struct {
	Protocol map[layers.IPProtocol]struct{}
	IP       []*net.IPNet
	Port     []*Uint16Range
}

func NewFilter(protocol []string, ip []string, port []string) (*Filter, error) {
	filter := &Filter{
		Protocol: make(map[layers.IPProtocol]struct{}),
		IP:       nil,
		Port:     nil,
	}
	for _, p := range protocol {
		switch strings.ToLower(p) {
		case "tcp":
			filter.Protocol[layers.IPProtocolTCP] = struct{}{}
		case "udp":
			filter.Protocol[layers.IPProtocolUDP] = struct{}{}
		}
	}
	if len(ip) != 0 {
		filter.IP = make([]*net.IPNet, 0)
		for _, s := range ip {
			_, ipNet, err := net.ParseCIDR(s)
			if err != nil {
				return nil, fmt.Errorf("failed to parse ip cidr filter string, %v", err)
			}
			filter.IP = append(filter.IP, ipNet)
		}
	}
	if len(port) != 0 {
		filter.Port = make([]*Uint16Range, 0)
		for _, s := range port {
			uint16Range, err := ParseUint16Range(s)
			if err != nil {
				return nil, fmt.Errorf("failed to parse port filter string, %v", err)
			}
			filter.Port = append(filter.Port, uint16Range)
		}
	}
	return filter, nil
}
