package main

import (
	"flag"
	"io"
	"log"
	"net"
	"os"
	"os/exec"
	"runtime"
	"strings"
	"sync"
	"sync/atomic"
	"time"
)

const UDPChanLen = 4096                          // number of messages to queue up from udp, helps buffer out spikes
const NumPacketWorkers = 4                       // number of goroutines to consume sflow and emit structured packet data
const PacketWorkerChanLen = NumPacketWorkers * 4 // number of structured packets to queue up for aggregation
const ReportInterval = time.Second * 60

func main() {
	log.SetFlags(0) // remove date/time from stdlogs (logger gives us those)

	var listenAddrs string
	var filterProtocol string
	var filterIP string
	var filterPort string
	var logstashAddr string
	var logstashProtocol string
	var numProcs int
	flag.StringVar(&listenAddrs, "listen-addr", "127.0.0.1:9876", "UDP IP:Port addresses to listen to for sflow packets (comma-separated)")
	flag.StringVar(&filterProtocol, "filter-protocol", "tcp", "Filter packets of these protocols (comma-separated, inclusive)")
	flag.StringVar(&filterIP, "filter-ip", "", "Filter packets whose IP Addresses match these CIDR net masks (comma-separated, inclusive, empty matches all)")
	flag.StringVar(&filterPort, "filter-port", "", "Filter packets whose port numbers match these ranges (comma-separated, inclusive, empty matches all)")
	flag.StringVar(&logstashAddr, "logstash-addr", "", "IP:Port address to send logstash-bound messages to (unset defaults to stdout)")
	flag.StringVar(&logstashProtocol, "logstash-protocol", "udp", "Networking protocol for logstash-addr (udp or tcp)")
	flag.IntVar(&numProcs, "num-procs", 1, "Number of OS threads to use for this daemon")
	flag.Parse()

	runtime.GOMAXPROCS(numProcs)
	usingProcs := runtime.GOMAXPROCS(numProcs) // this returns the number which was previously set, so call twice to get current number
	log.Printf("Starting sflow logger daemon with %d OS thread(s)", usingProcs)

	udpChan := make(chan []byte, UDPChanLen)
	var udpWorkerWG sync.WaitGroup
	for _, a := range strings.Split(listenAddrs, ",") {
		udpWorkerWG.Add(1)
		conn := listenUDP(a)
		log.Printf("Listening for sflows on %s", conn.LocalAddr().String())
		defer conn.Close()
		go receiveUDPWorker(conn, udpChan, udpWorkerWG)
	}
	go func(c chan []byte, wg sync.WaitGroup) {
		wg.Wait()
		close(c)
	}(udpChan, udpWorkerWG)

	var logstashWriter io.Writer
	logstashWriter = os.Stderr
	if logstashAddr != "" {
		conn, err := net.Dial(logstashProtocol, logstashAddr)
		if err != nil {
			log.Fatalf("failed to dial logstash service, %v", err)
		}
		defer conn.Close()
		logstashWriter = conn
	}
	logstashLogger = log.New(logstashWriter, "", 0)

	hostnameBytes, err := exec.Command("hostname", "-f").Output()
	if err != nil {
		log.Fatalf("failed to get local hostname, %v", err)
	}
	hostname = strings.TrimSpace(string(hostnameBytes))

	protocols := strings.Split(filterProtocol, ",")
	ips := []string{}
	ports := []string{}
	if filterIP != "" {
		ips = strings.Split(filterIP, ",")
	}
	if filterPort != "" {
		ports = strings.Split(filterPort, ",")
	}
	filter, err := NewFilter(protocols, ips, ports)
	if err != nil {
		log.Fatalf("failed to build packet filter, %v", err)
	}

	packetWorkerChan := make(chan *CapturedPacket, PacketWorkerChanLen)
	var packetWorkerWG sync.WaitGroup
	for i := 0; i < NumPacketWorkers; i++ {
		packetWorkerWG.Add(1)
		go packetWorker(udpChan, packetWorkerChan, packetWorkerWG, filter)
	}
	go func(c chan *CapturedPacket, wg sync.WaitGroup) {
		wg.Wait()
		close(c)
	}(packetWorkerChan, packetWorkerWG)

	aggregator := NewAggregator()
	t := make(chan struct{}, 0)
	go tick(ReportInterval, t)
	for {
		select {
		case capturedPacket, ok := <-packetWorkerChan:
			if !ok {
				return // worker channel closed - done
			}
			aggregator.Add(capturedPacket)
		case <-t:
			go aggregator.ForEach(logCapturedPacketDigest)
			aggregator = NewAggregator()
			log.Printf("Starting new interval, dropped %d packets", atomic.LoadUint64(&packetLoss))
		}
	}
}
