#!/usr/bin/env python2.7
import sys
import os
import subprocess

import boto.utils

USAGE="./{} mount_point"
FS_TOOL="mkfs.xfs"

def is_ebs_vol(ref):
    return ref.startswith("ebs")


def get_block_devices():
    return boto.utils.get_instance_metadata()['block-device-mapping']


def get_ebs_block_devices(device_map):
    return [d for r, d in device_map.items() if is_ebs_vol(r)]


def qualify_name(d):
    """
    Returns a fully qualified path to the block device, or None if one cannot be
    found

    http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-attaching-volume.html

    Asgard requests our devices be mapped under /dev/sdX, however the driver
    is permitted to remap those as it sees fit, which mean they can end up in the
    /dev/xvdX space
    """
    if not d:
        return None

    # check for presence in /dev/sdX
    qualified = '/dev/{}'.format(d)
    if os.path.exists(qualified):
        return qualified

    # check for presences in /dev/xvdX
    qualified = '/dev/xvd{}'.format(d[-1])
    if os.path.exists(qualified):
        return qualified

    return None


def get_qualified_names(devices):
    return map(qualify_name, devices)


def _call(*args):
    return subprocess.check_call(args)


def mkfs(devices):
    for d in devices:
        _call(FS_TOOL, "-f", d)

def mount(devices, mount_point):
    for d in devices:
        full_mount_point = '{}/{}'.format(mount_point, d[-1])
        _call("mkdir", "-p", full_mount_point)
        _call("mount", d, full_mount_point)


if __name__ == '__main__':
    if len(sys.argv)-1 != 1:
        sys.exit(USAGE.format(sys.argv[0]))

    mount_point = sys.argv[1]

    devices = get_ebs_block_devices(get_block_devices())
    devices = get_qualified_names(devices)
    mkfs(devices)
    mount(devices, mount_point)
