# Should be moved to a base module
provider "aws" {
	region = "us-west-2"
}

###################
# Security Groups #
###################

resource "aws_security_group" "ids-management" {
	name = "ids-${var.enviroment.identifier}management"
	description = "only allow access to port 22 from the rest of twitch"
	vpc_id = "${var.ids.vpc}"

	ingress {
		from_port = 22
		to_port = 22
		protocol = "tcp"
		cidr_blocks = [
		 	"10.0.0.0/8",
			"192.16.64.0/21",
			"199.9.248.0/21",
			"185.42.204.0/22",
			"33.99.10.96/29",
			"127.0.0.1/32",
			"38.99.10.97/32",
		]
    }
}

resource "aws_security_group" "ids-elk" {
    name = "ids-${var.enviroment.identifier}elk"
    description = "lock the elk cluster up"
	vpc_id="${var.ids.vpc}"

	ingress {
		from_port = 0
		to_port = 65535
		protocol = "tcp"
		security_groups = ["${aws_security_group.ids-logstash-ingest.id}"]
	}

	ingress {
		from_port = 0
		to_port = 65535
		protocol = "tcp"
		self = true
	}

	ingress {
		from_port = 0
		to_port = 65535
		protocol = "udp"
		self = true
	}

	ingress {
		from_port = 1028
		to_port = 1028
		protocol = "tcp"
		cidr_blocks = [
		 	"10.0.0.0/8",
			"192.16.64.0/21",
			"199.9.248.0/21",
			"185.42.204.0/22",
			"33.99.10.96/29",
			"127.0.0.1/32",
			"38.99.10.97/32",
		]
	}

	ingress {
		from_port = 10000
		to_port = 10000
		protocol = "tcp"
		cidr_blocks = [
		 	"10.0.0.0/8",
			"192.16.64.0/21",
			"199.9.248.0/21",
			"185.42.204.0/22",
			"33.99.10.96/29",
			"127.0.0.1/32",
			"38.99.10.97/32",
		]
	}

	ingress {
		from_port = 10001
		to_port = 10001
		protocol = "tcp"
		cidr_blocks = [
		 	"10.0.0.0/8",
			"192.16.64.0/21",
			"199.9.248.0/21",
			"185.42.204.0/22",
			"33.99.10.96/29",
			"127.0.0.1/32",
			"38.99.10.97/32",
		]
	}

	ingress {
		from_port = 514
		to_port = 514
		protocol = "udp"
		cidr_blocks = [
		 	"10.0.0.0/8",
			"192.16.64.0/21",
			"199.9.248.0/21",
			"185.42.204.0/22",
			"33.99.10.96/29",
			"127.0.0.1/32",
			"38.99.10.97/32",
		]
	}
}

resource "aws_security_group" "ids-logstash-ingest" {
	name = "ids-${var.enviroment.identifier}logstash-ingest"
	description = "allow internal hosts to push logs"
	vpc_id="${var.ids.vpc}"

	ingress {
		from_port = 0
		to_port = 65535
		protocol = "tcp"
		cidr_blocks = [
		 	"10.0.0.0/8",
			"192.16.64.0/21",
			"199.9.248.0/21",
			"185.42.204.0/22",
			"33.99.10.96/29",
			"127.0.0.1/32",
			"38.99.10.97/32",
		]
    }
}

resource "aws_security_group" "ids-dev" {
	name = "ids-${var.enviroment.identifier}dev"
	description = "free access from all twitch blocks"
	vpc_id="${var.ids.vpc}"

	ingress {
		from_port = 0
		to_port = 65535
		protocol = "tcp"
		cidr_blocks = [
		 	"10.0.0.0/8",
			"192.16.64.0/21",
			"199.9.248.0/21",
			"185.42.204.0/22",
			"33.99.10.96/29",
			"127.0.0.1/32",
			"38.99.10.97/32",
		]
    }
}

###################
# logstash ingest #
###################

resource "aws_route53_record" "ids-logstash-ingest" {
   zone_id = "${var.ids.privateR53}"
   name = "ids-${var.enviroment.identifier}logstash-ingest.${var.enviroment.domain}"
   type = "A"
   ttl = "300"
   records = ["${aws_instance.ids-logstash-ingest.*.private_ip}"]
}

resource "aws_elb" "ids-logstash-ingest" {
  name = "ids-${var.enviroment.identifier}logstash-ingest"
  subnets = ["${var.avail.zone0}", "${var.avail.zone1}", "${var.avail.zone2}"]

  security_groups = ["${aws_security_group.ids-logstash-ingest.id}", "${aws_security_group.ids-elk.id}"]
  internal = true

  listener {
    instance_port = 1028
    instance_protocol = "tcp"
    lb_port = 1028
    lb_protocol = "tcp"
  }

  listener {
  	instance_port = 10000
  	instance_protocol = "tcp"
  	lb_port = 10000
  	lb_protocol = "tcp"
  }

  listener {
  	instance_port = 10001
  	instance_protocol = "tcp"
  	lb_port = 10001
  	lb_protocol = "tcp"
  }

  health_check {
    healthy_threshold = 2
    unhealthy_threshold = 2
    timeout = 3
    target = "TCP:1028"
    interval = 5
  }

  instances = ["${aws_instance.ids-logstash-ingest.*.id}"]
  cross_zone_load_balancing = true
}

resource "aws_instance" "ids-logstash-ingest" {
	count = "${var.counts.ids-logstash-ingest}"
	ami = "ami-29ebb519"
	instance_type = "c4.large"
	subnet_id = "${lookup(var.avail, concat("zone", count.index))}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	iam_instance_profile = "SplashMountainLogstash"

	tags {
		Name = "ids-${var.enviroment.identifier}logstash-ingest-${count.index}.${var.enviroment.domain}"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-logstash-ingest ${self.private_ip}"
	}
}

#################
# elasticsearch #
#################

resource "aws_route53_record" "ids-logstash-es-head" {
   zone_id = "${var.ids.privateR53}"
   name = "ids-${var.enviroment.identifier}elasticsearch-head.${var.enviroment.domain}"
   type = "A"
   ttl = "300"
   records = ["${aws_instance.ids-elasticsearch-instancemount.0.private_ip}"]
}

resource "aws_instance" "ids-elasticsearch-ebsmount" {
	count = "${var.counts.ids-elasticsearch-ebsmount}"
	ami = "${var.ids.ebsmount-ami}"
	instance_type = "c4.4xlarge"
	subnet_id = "${var.avail.zone2}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	ebs_optimized = true
	ebs_block_device {
		device_name = "/dev/sdb"
		volume_type = "gp2"
		volume_size = 1024
		delete_on_termination = true
	}

	ebs_block_device {
		device_name = "/dev/sdc"
		volume_type = "gp2"
		volume_size = 1024
		delete_on_termination = true
	}

	ebs_block_device {
		device_name = "/dev/sdd"
		volume_type = "gp2"
		volume_size = 1024
		delete_on_termination = true
	}

	ebs_block_device {
		device_name = "/dev/sde"
		volume_type = "gp2"
		volume_size = 1024
		delete_on_termination = true
	}

	tags {
		Name = "ids-${var.enviroment.identifier}elasticsearch-ebsmount-${count.index}.${var.enviroment.domain}"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-elasticsearch-ebsmount ${self.private_ip}"
	}
}

resource "aws_instance" "ids-elasticsearch-instancemount" {
	count = "${var.counts.ids-elasticsearch-instancemount}"
	ami = "${var.ids.pv-ami}"
	instance_type = "m1.xlarge"
	subnet_id = "${var.avail.zone2}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	ephemeral_block_device {
		device_name = "/dev/sdb"
		virtual_name = "ephemeral0"
	}

	ephemeral_block_device {
		device_name = "/dev/sdc"
		virtual_name = "ephemeral1"
	}

	ephemeral_block_device {
		device_name = "/dev/sdd"
		virtual_name = "ephemeral2"
	}

	ephemeral_block_device {
		device_name = "/dev/sde"
		virtual_name = "ephemeral3"
	}

	tags {
		Name = "ids-${var.enviroment.identifier}elasticsearch-instancemount-${count.index}.${var.enviroment.domain}"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-elasticsearch-instancemount ${self.private_ip}"
	}
}


##########
# kibana #
##########

resource "aws_security_group" "ids-kibana-frontend" {
	name = "ids-${var.enviroment.identifier}kibana-frontend"
	description = "allows port 80 and 443 access to the ids kibana frontend"
	vpc_id = "${var.ids.vpc}"

	ingress {
		from_port = 80
		to_port = 80
		protocol = "tcp"
		cidr_blocks = [
		 	"10.0.0.0/8",
			"192.16.64.0/21",
			"199.9.248.0/21",
			"185.42.204.0/22",
			"33.99.10.96/29",
			"127.0.0.1/32",
			"38.99.10.97/32",
		]
    }

	ingress {
		from_port = 443
		to_port = 443
		protocol = "tcp"
		cidr_blocks = [
		 	"10.0.0.0/8",
			"192.16.64.0/21",
			"199.9.248.0/21",
			"185.42.204.0/22",
			"33.99.10.96/29",
			"127.0.0.1/32",
			"38.99.10.97/32",
		]
    }
}

resource "aws_route53_record" "ids-kibana" {
   zone_id = "${var.ids.privateR53}"
   name = "ids-${var.enviroment.identifier}kibana.${var.enviroment.domain}"
   type = "CNAME"
   ttl = "300"
   records = ["${aws_elb.ids-frontend.dns_name}"]
}

resource "aws_route53_record" "ids-elasticsearch-status" {
	zone_id = "${var.ids.privateR53}"
	name = "ids-${var.enviroment.identifier}elasticsearch-status.${var.enviroment.domain}"
	type = "CNAME"
	ttl = "300"
	records = ["${aws_elb.ids-frontend.dns_name}"]
}

resource "aws_elb" "ids-frontend" {
	name = "ids-${var.enviroment.identifier}kibana-frontend"
	subnets = ["${var.avail.zone0}", "${var.avail.zone1}", "${var.avail.zone2}"]

	security_groups = ["${aws_security_group.ids-kibana-frontend.id}", "${aws_security_group.ids-elk.id}"]
	internal = true

	listener {
		instance_port = 80
		instance_protocol = "http"
		lb_port = 80
		lb_protocol = "http"
	}

	listener {
		instance_port = 8080
		instance_protocol = "http"
		lb_port = 443
		lb_protocol = "https"
		ssl_certificate_id = "${var.ids.ids-kibana-cert}"
	}

	health_check {
		healthy_threshold = 2
		unhealthy_threshold = 2
		timeout = 3
		target = "HTTP:8081/"
		interval = 5
	}

	instances = ["${aws_instance.ids-kibana.*.id}"]
	cross_zone_load_balancing = true
}

resource "aws_instance" "ids-kibana" {
	count = "${var.counts.ids-kibana}"
	ami = "${var.ids.base-ami}"
	instance_type = "t2.small"
	subnet_id = "${lookup(var.avail, concat("zone", count.index))}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	tags {
		Name = "ids-${var.enviroment.identifier}kibana-${count.index}.${var.enviroment.domain}"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-kibana ${self.private_ip}"
	}
}

resource "aws_route53_record" "ids-logstash-syslog" {
   zone_id = "${var.ids.privateR53}"
   name = "ids-${var.enviroment.identifier}logstash-syslog.${var.enviroment.domain}"
   type = "A"
   ttl = "300"
   records = ["${aws_instance.ids-logstash-syslog.*.private_ip}"]
}

resource "aws_instance" "ids-logstash-syslog" {
	count = "${var.counts.ids-logstash-syslog}"
	ami = "ami-29ebb519"
	instance_type = "c4.8xlarge"
	subnet_id = "${lookup(var.avail, concat("zone", count.index))}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	tags {
		Name = "ids-${var.enviroment.identifier}logstash-syslog-${count.index}.${var.enviroment.domain}"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-logstash-syslog ${self.private_ip}"
	}
}
