# Should be moved to a base module
provider "aws" {
	region = "us-west-2"
}

#####################
# Production Config #
#####################

module "twitch" {
	source = "../modules/twitch"
}

variable "amis" {
	default {
		base = "ami-bdb6948d"
		ebsmount = "ami-29e2ce19"
	}
}

variable "node_counts" {
	default {
		es_head = 2
		es_ebsmount = 7
		es_search = 1
		logstash = 12
	}
}

###################
# Security Groups #
###################

resource "aws_security_group" "ids-management" {
	name = "ids-management"
	description = "only allow access to port 22 from the rest of twitch"
	vpc_id = "${module.twitch.vpc}"

	ingress {
		from_port = 22
		to_port = 22
		protocol = "tcp"
		cidr_blocks = ["${split(",", module.twitch.networks)}"]
    }
}

resource "aws_security_group" "ids-elk" {
    name = "ids-elk"
    description = "lock the elk cluster up"
	vpc_id="${module.twitch.vpc}"

	ingress {
		from_port = 0
		to_port = 65535
		protocol = "tcp"
		security_groups = ["${aws_security_group.ids-logstash-ingest.id}"]
	}

	ingress {
		from_port = 0
		to_port = 65535
		protocol = "tcp"
		self = true
	}

	ingress {
		from_port = 0
		to_port = 65535
		protocol = "udp"
		self = true
	}

	ingress {
		from_port = 1028
		to_port = 1028
		protocol = "tcp"
		cidr_blocks = ["${split(",", module.twitch.networks)}"]
	}

	ingress {
		from_port = 10000
		to_port = 10000
		protocol = "tcp"
		cidr_blocks = ["${split(",", module.twitch.networks)}"]
	}

	ingress {
		from_port = 10001
		to_port = 10001
		protocol = "tcp"
		cidr_blocks = ["${split(",", module.twitch.networks)}"]
	}

	ingress {
		from_port = 10002
		to_port = 10002
		protocol = "tcp"
		cidr_blocks = ["${split(",", module.twitch.networks)}"]
	}

	ingress {
		from_port = 514
		to_port = 514
		protocol = "udp"
		cidr_blocks = ["${split(",", module.twitch.networks)}"]
	}
}

resource "aws_security_group" "ids-logstash-ingest" {
	name = "ids-logstash-ingest"
	description = "allow internal hosts to push logs"
	vpc_id="${module.twitch.vpc}"

	ingress {
		from_port = 0
		to_port = 65535
		protocol = "tcp"
		cidr_blocks = ["${split(",", module.twitch.networks)}"]
    }
}

resource "aws_security_group" "ids-dev" {
	name = "ids-dev"
	description = "free access from all twitch blocks"
	vpc_id="${module.twitch.vpc}"

	ingress {
		from_port = 0
		to_port = 65535
		protocol = "tcp"
		cidr_blocks = ["${split(",", module.twitch.networks)}"]
    }
}

#################
# elasticsearch #
#################

resource "aws_instance" "ids-elasticsearch-ebsmount" {
	count = "${var.node_counts.es_ebsmount}"
	ami = "${var.amis.ebsmount}"
	instance_type = "c4.4xlarge"
	subnet_id = "${element(split(",", module.twitch.subnets), 2)}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	ebs_optimized = true
	ebs_block_device {
		device_name = "/dev/sdb"
		volume_type = "gp2"
		volume_size = 1024
		delete_on_termination = true
	}

	ebs_block_device {
		device_name = "/dev/sdc"
		volume_type = "gp2"
		volume_size = 1024
		delete_on_termination = true
	}

	ebs_block_device {
		device_name = "/dev/sdd"
		volume_type = "gp2"
		volume_size = 1024
		delete_on_termination = true
	}

	ebs_block_device {
		device_name = "/dev/sde"
		volume_type = "gp2"
		volume_size = 1024
		delete_on_termination = true
	}

	tags {
		Name = "ids-elasticsearch-ebsmount-${count.index}.prod.us-west2.justin.tv"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-elasticsearch-ebsmount ${self.private_ip}"
	}
}

######################
# elasticsearch head #
######################

resource "aws_route53_record" "ids-elasticsearch-head" {
	count = "${var.node_counts.es_head}"
	zone_id = "${module.twitch.privateR53}"
	name = "ids-elasticsearch-head-${count.index}.prod.us-west2.justin.tv"
	type = "A"
	ttl = "300"
	records = ["${element(aws_instance.ids-elasticsearch-head.*.private_ip, count.index)}"]
}

resource "aws_instance" "ids-elasticsearch-head" {
	count = "${var.node_counts.es_head}"
	ami = "${var.amis.base}"
	instance_type = "m3.medium"
	subnet_id = "${element(split(",", module.twitch.subnets), 2)}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	tags {
		Name = "ids-elasticsearch-head-${count.index}.prod.us-west2.justin.tv"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-elasticsearch-head ${self.private_ip}"
	}
}

########################
# elasticsearch search #
########################

resource "aws_route53_record" "ids-elasticsearch-search" {
	zone_id = "${module.twitch.privateR53}"
	name = "ids-elasticsearch-search.prod.us-west2.justin.tv"
	type = "A"
	ttl = "300"
	records = ["${aws_instance.ids-elasticsearch-search.*.private_ip}"]
}

resource "aws_instance" "ids-elasticsearch-search" {
	count = "${var.node_counts.es_search}"
	ami = "${var.amis.base}"
	instance_type = "m3.medium"
	subnet_id = "${element(split(",", module.twitch.subnets), 2)}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	tags {
		Name = "ids-elasticsearch-search-${count.index}.prod.us-west2.justin.tv"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-elasticsearch-search ${self.private_ip}"
	}
}

########################
# elasticsearch ingest #
########################

resource "aws_route53_record" "ids-elasticsearch-ingest" {
   zone_id = "${module.twitch.privateR53}"
   name = "ids-elasticsearch-ingest.prod.us-west2.justin.tv"
   type = "A"
   ttl = "300"
   records = ["${aws_instance.ids-elasticsearch-ingest.*.private_ip}"]
}

resource "aws_instance" "ids-elasticsearch-ingest" {
	count = 6
	ami = "ami-29ebb519"
	instance_type = "c4.large"
	subnet_id = "${element(split(",", module.twitch.subnets), count.index)}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	tags {
		Name = "ids-elasticsearch-ingest-${count.index}.prod.us-west2.justin.tv"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-elasticsearch-ingest ${self.private_ip}"
	}
}

##########
# kibana #
##########

resource "aws_security_group" "ids-kibana-frontend" {
	name = "ids-kibana-frontend"
	description = "allows port 80 and 443 access to the ids kibana frontend"
	vpc_id = "${module.twitch.vpc}"

	ingress {
		from_port = 80
		to_port = 80
		protocol = "tcp"
		cidr_blocks = [
		 	"10.0.0.0/8",
			"192.16.64.0/21",
			"199.9.248.0/21",
			"185.42.204.0/22",
			"33.99.10.96/29",
			"127.0.0.1/32",
			"38.99.10.97/32",
		]
    }

	ingress {
		from_port = 443
		to_port = 443
		protocol = "tcp"
		cidr_blocks = [
		 	"10.0.0.0/8",
			"192.16.64.0/21",
			"199.9.248.0/21",
			"185.42.204.0/22",
			"33.99.10.96/29",
			"127.0.0.1/32",
			"38.99.10.97/32",
		]
    }
}

resource "aws_route53_record" "ids-kibana" {
   zone_id = "${module.twitch.privateR53}"
   name = "ids-kibana.prod.us-west2.justin.tv"
   type = "CNAME"
   ttl = "300"
   records = ["${aws_elb.ids-frontend.dns_name}"]
}

resource "aws_route53_record" "ids-elasticsearch-status" {
	zone_id = "${module.twitch.privateR53}"
	name = "ids-elasticsearch-status.prod.us-west2.justin.tv"
	type = "CNAME"
	ttl = "300"
	records = ["${aws_elb.ids-frontend.dns_name}"]
}

resource "aws_elb" "ids-frontend" {
	name = "ids-kibana-frontend"
	subnets = ["${split(",", module.twitch.subnets)}"]

	security_groups = ["${aws_security_group.ids-kibana-frontend.id}", "${aws_security_group.ids-elk.id}"]
	internal = true

	listener {
		instance_port = 80
		instance_protocol = "http"
		lb_port = 80
		lb_protocol = "http"
	}

	listener {
		instance_port = 8080
		instance_protocol = "http"
		lb_port = 443
		lb_protocol = "https"
		ssl_certificate_id = "${module.twitch.jtv-ssl-cert}"
	}

	health_check {
		healthy_threshold = 2
		unhealthy_threshold = 2
		timeout = 3
		target = "HTTP:8081/"
		interval = 5
	}

	instances = ["${aws_instance.ids-kibana.*.id}"]
	cross_zone_load_balancing = true
}

resource "aws_instance" "ids-kibana" {
	count = 1
	ami = "${var.amis.base}"
	instance_type = "t2.small"
	subnet_id = "${element(split(",", module.twitch.subnets), count.index)}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	tags {
		Name = "ids-kibana-${count.index}.prod.us-west2.justin.tv"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-kibana ${self.private_ip}"
	}
}

#################
# syslog ingest #
#################

# this section is named poorly because originally we ran logstash
# but had to switch to a go program due to performance limitations

resource "aws_route53_record" "ids-logstash-syslog" {
   zone_id = "${module.twitch.privateR53}"
   name = "ids-logstash-syslog.prod.us-west2.justin.tv"
   type = "A"
   ttl = "300"
   records = ["${aws_instance.ids-logstash-syslog.*.private_ip}"]
}

resource "aws_instance" "ids-logstash-syslog" {
	count = 4
	ami = "ami-29ebb519"
	instance_type = "c4.8xlarge"
	subnet_id = "${element(split(",", module.twitch.subnets), count.index)}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	tags {
		Name = "ids-logstash-syslog-${count.index}.prod.us-west2.justin.tv"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-logstash-syslog ${self.private_ip}"
	}
}

##################
# logstash proxy #
##################

resource "aws_elb" "ids-logstashproxy" {
  name = "ids-logstashproxy"
  subnets = ["${split(",", module.twitch.subnets)}"]

  security_groups = ["${aws_security_group.ids-logstash-ingest.id}", "${aws_security_group.ids-elk.id}"]
  internal = true

  listener {
  	instance_port = 10000
  	instance_protocol = "tcp"
  	lb_port = 10000
  	lb_protocol = "tcp"
  }

  listener {
  	instance_port = 10001
  	instance_protocol = "tcp"
  	lb_port = 10001
  	lb_protocol = "tcp"
  }

  listener {
  	instance_port = 10002
  	instance_protocol = "tcp"
  	lb_port = 10002
  	lb_protocol = "tcp"
  }

  listener {
  	instance_port = 10003
  	instance_protocol = "tcp"
  	lb_port = 10003
  	lb_protocol = "tcp"
  }

  health_check {
    healthy_threshold = 2
    unhealthy_threshold = 2
    timeout = 3
    target = "TCP:1028"
    interval = 5
  }

  instances = ["${aws_instance.ids-logstashproxy.*.id}"]
  cross_zone_load_balancing = true
}

resource "aws_route53_record" "ids-logstash-ingest" {
   zone_id = "${module.twitch.privateR53}"
   name = "ids-logstash-ingest.prod.us-west2.justin.tv"
   type = "A"
   ttl = "300"
   records = ["${aws_instance.ids-logstashproxy.*.private_ip}"]
}

resource "aws_instance" "ids-logstashproxy" {
	count = 6
	ami = "${var.amis.base}"
	instance_type = "m3.xlarge"
	subnet_id = "${element(split(",", module.twitch.subnets), count.index)}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	iam_instance_profile = "SplashMountainLogstash"

	tags {
		Name = "ids-logstashproxy-${count.index}.prod.us-west2.justin.tv"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-logstashproxy ${self.private_ip}"
	}
}

#############
# zookeeper #
#############

resource "aws_route53_record" "ids-zookeeper" {
	count = 0
	zone_id = "${module.twitch.privateR53}"
	name = "ids-zookeeper-${count.index}.prod.us-west2.justin.tv"
	type = "A"
	ttl = "300"
	records = ["${element(aws_instance.ids-zookeeper.*.private_ip, count.index)}"]
}

resource "aws_instance" "ids-zookeeper" {
	count = 0
	ami = "${var.amis.base}"
	instance_type = "m3.large"
	subnet_id = "${element(split(",", module.twitch.subnets), count.index)}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	iam_instance_profile = "IdsZookeeper"

	tags {
		Name = "ids-zookeeper-${count.index}.prod.us-west2.justin.tv"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-zookeeper ${self.private_ip}"
	}

	user_data = <<END_OF_USERDATA
#cloud-config
hostname: ids-zookeeper-${count.index}
fqdn: ids-zookeeper-${count.index}.prod.us-west2.justin.tv
END_OF_USERDATA
}

#########
# kafka #
#########

resource "aws_route53_record" "ids-kafka" {
	count = 0
	zone_id = "${module.twitch.privateR53}"
	name = "ids-kafka-${count.index}.prod.us-west2.justin.tv"
	type = "A"
	ttl = "300"
	records = ["${element(aws_instance.ids-kafka.*.private_ip, count.index)}"]
}

resource "aws_instance" "ids-kafka" {
	count = 0
	ami = "${var.amis.base}"
	instance_type = "d2.xlarge"
	subnet_id = "${element(split(",", module.twitch.subnets), count.index)}"
	security_groups = ["${aws_security_group.ids-elk.id}", "${aws_security_group.ids-management.id}"]
	key_name = "ids_management"

	ephemeral_block_device {
		device_name = "/dev/sdb"
		virtual_name = "ephemeral0"
	}

	ephemeral_block_device {
		device_name = "/dev/sdc"
		virtual_name = "ephemeral1"
	}

	ephemeral_block_device {
		device_name = "/dev/sdd"
		virtual_name = "ephemeral2"
	}

	tags {
		Name = "ids-kafka-${count.index}.prod.us-west2.justin.tv"
	}

	provisioner "local-exec" {
		command = "../scripts/provision.sh ids-kafka ${self.private_ip}"
	}

	user_data = <<END_OF_USERDATA
#cloud-config
hostname: ids-kafka-${count.index}
fqdn: ids-kafka-${count.index}.prod.us-west2.justin.tv
END_OF_USERDATA
}
