package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"github.com/jeromer/syslogparser"
	"github.com/jeromer/syslogparser/rfc3164"
	"log"
	"net"
	"runtime"
	"sync"
	"sync/atomic"
	"time"
)

var workers = flag.Int("workers", 1, "how many workers to spawn")
var addr = flag.String("addr", "ids-logstash-ingest.prod.us-west2.justin.tv:10003", "logstash connect address")

var failCount int64
var dropCount int64
var parseCount int64
var shipCount int64

var bufPool = &sync.Pool{New: newBuf}

func newBuf() interface{} {
	return make([]byte, 4096)
}

func getBuf() []byte {
	return bufPool.Get().([]byte)
}

func putBuf(buf []byte) {
	if cap(buf) != 4096 {
		log.Fatalf("returning invalid buf")
	}

	bufPool.Put(buf[:cap(buf)])
}

func mustListenUDP(addr string) *net.UDPConn {
	udpAddr, err := net.ResolveUDPAddr("udp", addr)
	if err != nil {
		log.Fatal(err)
	}

	conn, err := net.ListenUDP("udp", udpAddr)
	if err != nil {
		log.Fatal(err)
	}

	return conn
}

func acceptPackets(out chan []byte, in *net.UDPConn) {
	for {
		buf := getBuf()
		n, err := in.Read(buf)
		if err != nil {
			log.Fatal(err)
		}

		select {
		case out <- buf[:n]:
		default:
			atomic.AddInt64(&dropCount, 1)
		}
	}
}

type Packet struct {
	Parts syslogparser.LogParts
	Raw   []byte
}

func parse(buf []byte) (ret syslogparser.LogParts, err error) {
	defer func() {
		if err := recover(); err != nil {
			err = fmt.Errorf("syslogparser panic")
		}
	}()

	parser := rfc3164.NewParser(buf)
	if err := parser.Parse(); err != nil {
		return nil, err
	}

	return parser.Dump(), nil
}

func parsePackets(out chan *Packet, in chan []byte) {
	for buf := range in {
		parts, err := parse(buf)
		if err != nil {
			atomic.AddInt64(&failCount, 1)
		} else {
			pkt := &Packet{
				Parts: parts,
				Raw:   buf,
			}
			select {
			case out <- pkt:
			default:
				atomic.AddInt64(&dropCount, 1)
			}

			atomic.AddInt64(&parseCount, 1)
		}
	}
}

// func (p *Parser) Dump() syslogparser.LogParts {
// 	return syslogparser.LogParts{
// 		"timestamp": p.header.timestamp,
// 		"hostname":  p.header.hostname,
// 		"tag":       p.message.tag,
// 		"content":   p.message.content,
// 		"priority":  p.priority.P,
// 		"facility":  p.priority.F.Value,
// 		"severity":  p.priority.S.Value,
// 	}
// }

type LogstashSyslog struct {
	Message   string `json:"message"`
	Hostname  string `json:"syslog_hostname"`
	Content   string `json:"syslog_content"`
	Program   string `json:"syslog_program"`
	Timestamp string `json:"syslog_timestamp"`
	Type      string `json:"type"`
	Shipper   string `json:"shipper"`
}

func shipPackets(addr string, in chan *Packet) {
	for {
		conn, err := net.DialTimeout("tcp", addr, 5*time.Second)
		if err != nil {
			log.Printf("failed to dial %s: %s", addr, err)
			continue
		}

		encoder := json.NewEncoder(conn)
		for pkt := range in {
			hostname, _ := pkt.Parts["hostname"].(string)
			content, _ := pkt.Parts["content"].(string)
			program, _ := pkt.Parts["tag"].(string)
			timestamp, _ := pkt.Parts["timestamp"].(time.Time)
			err := encoder.Encode(LogstashSyslog{
				Message:   string(pkt.Raw),
				Hostname:  hostname,
				Content:   content,
				Program:   program,
				Timestamp: timestamp.String(),
				Type:      "syslog",
				Shipper:   "syslogingest",
			})

			if err != nil {
				log.Printf("failed to ship packet: %v", pkt)
				conn.Close()
				break
			}

			putBuf(pkt.Raw)
			atomic.AddInt64(&shipCount, 1)
		}
	}
}

func main() {
	log.SetFlags(0)
	flag.Parse()

	runtime.GOMAXPROCS(*workers)

	conn := mustListenUDP(":514")

	in := make(chan []byte, 262144)
	parsed := make(chan *Packet, 262144)

	for i := 0; i < *workers; i++ {
		go acceptPackets(in, conn)
		go parsePackets(parsed, in)
		go shipPackets(*addr, parsed)

	}

	var lastDrop, lastFail, lastParse, lastShip int64
	for _ = range time.Tick(1 * time.Second) {
		drop := atomic.LoadInt64(&dropCount)
		fail := atomic.LoadInt64(&failCount)
		parse := atomic.LoadInt64(&parseCount)
		ship := atomic.LoadInt64(&shipCount)

		log.Printf("drops: %d, fails: %d, parsed: %d, shipped: %d", drop-lastDrop, fail-lastFail, parse-lastParse, ship-lastShip)

		lastDrop = drop
		lastFail = fail
		lastParse = parse
		lastShip = ship
	}
}
