package jwt_test

import (
	"code.justin.tv/infosec/jwt"
	"code.justin.tv/infosec/jwt/claim"
	"crypto/rand"
	"encoding/json"
	"fmt"
	"os"
	"time"
)

type Claims struct {
	// here we're using the claim package to verify that
	// our token isn't invlid with regard to time.
	Expires   claim.Exp `json:"exp"`
	NotBefore claim.Nbf `json:"nbf"`
	IssuedAt  claim.Iat `json:"iat"`

	Prn string `json:"prn"` // name of user

}

//IDToken is a token representing a signed claim on a user.
type IDToken struct {
	Claims
}

var (
	algorithm jwt.Algorithm
	header    jwt.Header
)

var secret = []byte("secret")

func init() {
	// generate secret if not present securely
	// secret will never be nil, since this an example.
	if secret == nil {
		if _, err := rand.Read(secret[:]); err != nil {
			panic(err)
		}
	}

	algorithm = jwt.HS512(secret[:])
	header = jwt.NewHeader(algorithm)
}

func (i IDToken) MarshalText() ([]byte, error) { return jwt.Encode(header, i.Claims, algorithm) }
func (i IDToken) String() string {
	b, err := i.MarshalText()
	if err != nil {
		return fmt.Sprint(err)
	}

	return string(b)
}

//UnmarshalBytes decodes and validates the JWT.
func (i *IDToken) UnmarshalBytes(b []byte) (err error) {
	var h jwt.Header
	if err = jwt.DecodeAndValidate(&h, i, algorithm, b); err != nil {
		return
	}

	if err = header.ValidateEqual(header); err != nil {
		return
	}

	if !claim.Validate(i) {
		return
	}

	return
}

//Here, we demonstrate encoding and decoding of JSON Web Tokens using symmetric (HMAC) crypto.
//In this case, only parties which can generate tokens can verify them.
func Example_hmac() {

	//you would use time.Now() here but I'm using a constant date for testing purposes
	joe := IDToken{
		Claims{
			//example will fail in the year 3000, sorry :(
			Expires:   claim.Exp(time.Date(3000, 12, 17, 0, 0, 0, 0, time.UTC)),
			NotBefore: claim.Nbf(time.Date(1993, 12, 17, 0, 0, 0, 0, time.UTC)),
			IssuedAt:  claim.Iat(time.Date(2015, 12, 17, 0, 0, 0, 0, time.UTC)),
			Prn:       "joe",
		},
	}

	//encode a jwt to stdout:
	if _, err := fmt.Println(joe); err != nil {
		return
	}

	j := json.NewEncoder(os.Stdout)

	var tok IDToken

	// decode into IDToken
	if err := tok.UnmarshalBytes([]byte("eyJhbGciOiJIUzUxMiIsInR5cCI6IkpXVCJ9.eyJleHAiOjMyNTMzOTIwMDAwLCJuYmYiOjc1NjA4NjQwMCwiaWF0IjoxNDUwMzEwNDAwLCJwcm4iOiJqb2UifQ==.6NIu8h190VZ9AwIE0JJSaKoniBzPf1_-059N285FD7crWpjUhnqwDd18WUbiUC9elYGuVJVPWfDu117TRvuI8w==")); err != nil {
		fmt.Println(err)
	}

	j.Encode(tok.Claims)

	// Output:
	// eyJhbGciOiJIUzUxMiIsInR5cCI6IkpXVCJ9.eyJleHAiOjMyNTMzOTIwMDAwLCJuYmYiOjc1NjA4NjQwMCwiaWF0IjoxNDUwMzEwNDAwLCJwcm4iOiJqb2UifQ==.6NIu8h190VZ9AwIE0JJSaKoniBzPf1_-059N285FD7crWpjUhnqwDd18WUbiUC9elYGuVJVPWfDu117TRvuI8w==
	// {"exp":32533920000,"nbf":756086400,"iat":1450310400,"prn":"joe"}
}
