package jwt_test

import (
	"code.justin.tv/infosec/jwt"
	"code.justin.tv/infosec/jwt/claim"
	"crypto/rsa"
	"encoding/json"
	"fmt"
	"os"
	"time"
)

type TokenClaims struct {
	Expires   claim.Exp `json:"exp"`
	NotBefore claim.Nbf `json:"nbf"`
	IssuedAt  claim.Iat `json:"iat"`

	Prn string `json:"prn"` // name of user

}

//UserToken is a token representing a signed claim on a user.
type UserToken struct {
	TokenClaims
}

var alg jwt.Algorithm
var jwtHeader jwt.Header

var pkey *rsa.PrivateKey

var validateAlg jwt.Algorithm

func init() {
	// ssh-keygen -t rsa -f _key/id_rsa.pem -m pem
	var err error
	//example key, don't actually use this one!
	//see the example for RSAValidateOnly for how to do this with
	//just a public key.
	pkey, err = jwt.ReadRSAPrivateKey("_key/id_rsa.pem")
	if err != nil {
		panic(err)
	}

	alg = jwt.RS512(pkey)

	jwtHeader = jwt.NewHeader(alg)

	k, err := jwt.ReadRSAPublicKey("_key/id_rsa.pub.pem")
	if err != nil {
		panic(err)
	}

	validateAlg = jwt.RSAValidateOnly(jwt.RS512, k)
}

func (u UserToken) MarshalText() ([]byte, error) {
	return jwt.Encode(jwtHeader, u.TokenClaims, alg)
}

func (u UserToken) String() string {
	b, err := u.MarshalText()
	if err != nil {
		return fmt.Sprintf("failing jwt: %s", err)
	}

	return string(b)
}

//UnmarshalText decodes and validates the JWT.
func (i *UserToken) UnmarshalText(b []byte) (err error) {
	var h jwt.Header
	if err = jwt.DecodeAndValidate(&h, &i.TokenClaims, alg, b); err != nil {
		return
	}

	if err = jwtHeader.ValidateEqual(jwtHeader); err != nil {
		return
	}

	if !claim.Validate(i) {
		return
	}

	return
}

//Asymmetric (public key) crypto can be used. This allows third parties to verify our tokens using our public key.
func Example_rsa() {
	//Here is our example token, a UserToken representing a user ("joe"),
	//a signed authorization that expires in the year 3000 and must not
	//be used before the year 1993. We also set the issued at date to the year 1993.
	//You would use time.Now() here but I'm using a constant date for testing purposes.
	joe := UserToken{
		TokenClaims{
			//example will fail in the year 3000, sorry :(
			Expires:   claim.Exp(time.Date(3000, 12, 17, 0, 0, 0, 0, time.UTC)),
			NotBefore: claim.Nbf(time.Date(1993, 12, 17, 0, 0, 0, 0, time.UTC)),
			IssuedAt:  claim.Iat(time.Date(2015, 12, 17, 0, 0, 0, 0, time.UTC)),
			Prn:       "joe",
		},
	}

	//for pushing json to stdout
	jo := json.NewEncoder(os.Stdout)

	fmt.Println("Our original example token structure:")
	jo.Encode(joe.TokenClaims)

	fmt.Println("Original token structure encoded as a JWT:")
	if _, err := fmt.Println(joe); err != nil {
		panic(err)
	}

	//Here, we create a new UserToken and fill in the
	//structure using our generated JWT.
	var tok UserToken

	btT, err := joe.MarshalText()
	if err != nil {
		panic(err)
	}

	//decode into UserToken
	if err := tok.UnmarshalText(btT); err != nil {
		panic(err)
	}

	fmt.Println("That JWT, now decoded into a new structure using an rsa.PrivateKey:")
	jo.Encode(tok.TokenClaims)

	// alternately, we can validate a JWT using public key
	var tok2 UserToken

	alg = validateAlg
	if err := tok2.UnmarshalText(btT); err != nil {
		panic(err)
	}

	fmt.Println("The JWT, now decoded into a new structure using an rsa.PublicKey:")
	jo.Encode(tok2.TokenClaims)

	// Output:
	//Our original example token structure:
	//{"exp":32533920000,"nbf":756086400,"iat":1450310400,"prn":"joe"}
	//Original token structure encoded as a JWT:
	//eyJhbGciOiJSUzUxMiIsInR5cCI6IkpXVCJ9.eyJleHAiOjMyNTMzOTIwMDAwLCJuYmYiOjc1NjA4NjQwMCwiaWF0IjoxNDUwMzEwNDAwLCJwcm4iOiJqb2UifQ==.1GxfvWrNMnr1bsbipymtCip-O5276T31yNv9yXtA5q9lJadIiPwlnFLwJq4oV7fhJawXQPQPuPCIN_Sz6eAxAZw69A0cdgCdeJ-s6Btv9M3PSxVv7CrJ12buAKB-epeySbNFcI8Ob0zRGMgoZnxVFagovNnpv_xbDKTTY1r35OydvrFrS6wGVf6gYA-pfsMn3v5Dx58zd9_xXNLgbHzPLSQDSeV9KfC_T6kEWukjiE3eN2ABWB8JskpZ95f4t_FAV578DCLiGGKmPNUlFOGWw3oe-soKrhU7kPzQq3KRKUB2kEF1KxHOw11rjWOPaJ23llGRg27tBjv05THDABc4Hw==
	//That JWT, now decoded into a new structure using an rsa.PrivateKey:
	//{"exp":32533920000,"nbf":756086400,"iat":1450310400,"prn":"joe"}
	//The JWT, now decoded into a new structure using an rsa.PublicKey:
	//{"exp":32533920000,"nbf":756086400,"iat":1450310400,"prn":"joe"}
}
