package api

import (
	"net/http"

	"code.justin.tv/insights/piper-service/internal/metrics"

	goji "goji.io"
	"goji.io/pat"

	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/identity/golibs/middleware"
	"code.justin.tv/insights/piper-service/backend/drops"
	"code.justin.tv/insights/piper-service/backend/extensions"
	"code.justin.tv/insights/piper-service/backend/games"
	"code.justin.tv/insights/piper-service/backend/mods"
	"code.justin.tv/insights/piper-service/backend/one_pager"
	"code.justin.tv/insights/piper-service/internal/clients/rbac"
	piper "code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/sse/malachai/pkg/s2s/callee"
)

type Server struct {
	*goji.Mux
	piperConfig       piper.PiperConfig
	gamesBackend      games.Backend
	extensionsBackend extensions.Backend
	onePagerBackend   one_pager.Backend
	modsBackend       mods.Backend
	dropsBackend      drops.Backend
	rbacClient        rbac.Client
}

func NewServer(gb games.Backend, eb extensions.Backend, db drops.Backend, op one_pager.Backend, mb mods.Backend, rbacClient rbac.Client, calleeClient callee.ClientAPI, piperConfig piper.PiperConfig) (*Server, error) {
	mux := twitchserver.NewServer()

	mux.Use(middleware.Source())
	mux.Use(middleware.Stats(metrics.GetTelemetryStatter(), 1))
	if calleeClient != nil {
		mux.Use(func(h http.Handler) http.Handler {
			return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				calleeClient.CapabilitiesInjectorMiddleware(h).ServeHTTP(w, r)
			})
		})
	}

	s := &Server{
		Mux:               mux,
		piperConfig:       piperConfig,
		gamesBackend:      gb,
		extensionsBackend: eb,
		dropsBackend:      db,
		onePagerBackend:   op,
		modsBackend:       mb,
		rbacClient:        rbacClient,
	}

	// Root
	s.HandleFunc(pat.Get("/"), createhandler(showHome))

	//
	// Internal endpoints (1st party)
	// ------------------------------

	// All available game reports for a user.
	// Returns a list of game ids that the user has access to and that has at least one generated report.
	s.HandleFunc(pat.Get("/report/games/:userid"), createhandler(s.GetGameReport))

	// Report type for a game if the user has access to it (if available, it is always overview_v2).
	// NOTE: only used from devsite (1st pary) through kraken/v5 (visage) to decide what report type to download.
	s.HandleFunc(pat.Get("/report/games/:userid/:game"), createhandler(s.GetGameReportTypes))

	// Presigns a URL to download a CSV report for a game, the URL expires in 5 minutes.
	// The report is dynamically generated with start_date and end_date, which by default is the retention period: from 2018-01-31 until today
	s.HandleFunc(pat.Get("/report/games/:userid/:game/:report"), createhandler(s.GetGameReportInstancesOrByDates))
	s.HandleFunc(pat.Get("/report/games/:userid/:game/:report/latest"), createhandler(s.GetGameReportInstancesOrByDates))

	// Report type for an extension_id (Client-ID) if the user has access to it (if available, it is always overview_v2).
	// NOTE: only used from devsite (1st pary) through kraken/v5 (visage) to decide what report type to download.
	s.HandleFunc(pat.Get("/report/extensions/:user_id/:extension_id"), createhandler(s.GetExtensionReportTypes))

	// Presigns a URL to download a CSV report for a game, the URL expires in 5 minutes.
	// The report is dynamically generated with start_date and end_date, which by default is the retention period: from 2018-01-31 until today
	s.HandleFunc(pat.Get("/report/extensions/:user_id/:extension_id/:report_type"), createhandler(s.GetExtensionReportInstancesOrByDates))
	s.HandleFunc(pat.Get("/report/extensions/:user_id/:extension_id/:report_type/latest"), createhandler(s.GetExtensionReportInstancesOrByDates))

	// All available mod reports for a user.
	// Returns a list of mod ids that the user has access to and that has at least one generated report.
	s.HandleFunc(pat.Get("/report/mods/:user_id"), createhandler(s.GetMods))

	// Presigns a URL to download a CSV report for a mod, the URL expires in 5 minutes.
	s.HandleFunc(pat.Get("/report/mods/:user_id/:mod_id/:report_type/latest"), createhandler(s.GetModReportLatest))

	// Presigns a URL to download a CSV drops report. The URL expires in 5 minutes.
	// Resource_id depends on report_type:
	//   "campaigns_overview": campaign_id
	//   "campaigns_top_streamers": campaign_id
	s.HandleFunc(pat.Get("/report/drops/:user_id/:resource_id/:report_type/latest"), createhandler(s.GetDropReportLatest))

	//
	// Endpoints for Helix (3rd party)
	// -------------------------------

	// Retrieve a filterable set of download URLs of games.
	s.HandleFunc(pat.Get("/helix/report/games/:userid"), createhandler(s.GetGameReportsForHelix))

	// Retrieve a filterable set of download URLs of extensions.
	s.HandleFunc(pat.Get("/helix/report/extensions/:userid"), createhandler(s.GetExtensionReportsForHelix))

	//
	// Endpoints for BoxArt Upload
	// -------------------------------

	// Request a pre signed S3 URL from Upload Service
	s.HandleFunc(pat.Get("/boxart/upload/:userid/:game"), createhandler(s.CreateUploadServiceRequest))

	//
	// Endpoints for Get One Pager Data
	// -------------------------------

	// Request a one pager data for a game
	s.HandleFunc(pat.Get("/onepager/games/:user_id/:game_id"), createhandler(s.GetOnePagerData))

	// Retrieve report download URL for one pager data for a game
	s.HandleFunc(pat.Get("/onepager/games/:user_id/:game_id/download"), createhandler(s.GetOnePagerDownloadURL))

	s.HandleFunc(pat.Get("/onepager/games/:user_id/:game_id/quarterstartrange"), createhandler(s.GetQuarterStartRange))

	return s, nil
}
