package api

import (
	"net/http"

	"code.justin.tv/insights/piper-service/lib/reports"
	"code.justin.tv/insights/piper-service/models"
	"goji.io/pat"
	"golang.org/x/net/context"
)

func showHome(ctx context.Context, r *http.Request) (interface{}, error) {
	return "Welcom to Piper", nil
}

func (s *Server) GetGameReport(ctx context.Context, r *http.Request) (interface{}, error) {
	userID := betterformat(pat.Param(r, "userid"))

	// Track get game report usage
	trackingEvent := models.SpadeEvent{
		UserID:       userID,
		ClientID:     r.Header.Get("Twitch-Client-ID"),
		IsFirstParty: true,
		Action:       "get_game_report",
		IfSucceed:    false,
	}

	games, err := s.getGameReport(ctx, userID)
	s.reportTracking(ctx, trackingEvent, trackingGamesBackend, err)
	return games, err
}

func (s *Server) getGameReport(ctx context.Context, userID string) (interface{}, error) {
	result, err := s.gamesBackend.GetGameReport(ctx, userID, s.piperConfig)

	if err != nil {
		return nil, err
	}

	return &models.GameReportsResp{Key: *result}, nil
}

func (s *Server) GetGameReportTypes(ctx context.Context, r *http.Request) (interface{}, error) {
	userID := betterformat(pat.Param(r, "userid"))
	gameID := betterformat(pat.Param(r, "game"))

	trackingEvent := models.SpadeEvent{
		UserID:       userID,
		DomainID:     gameID,
		ClientID:     r.Header.Get("Twitch-Client-ID"),
		IsFirstParty: true,
		Action:       "get_game_report_types",
		IfSucceed:    true,
	}

	reportTypes, err := s.getGameReportTypes(ctx, userID, gameID)
	// track staff get report types usage
	s.reportTracking(ctx, trackingEvent, trackingGamesBackend, err)

	return reportTypes, err
}

func (s *Server) getGameReportTypes(ctx context.Context, userID, gameID string) (interface{}, error) {
	ok, err := s.gamesBackend.UserCanAccess(ctx, "games", userID, gameID, reports.DevsiteGamesReportType)
	if !ok || err != nil {
		return "", err
	}

	result, err := s.gamesBackend.GetReportTypes(ctx, userID, gameID, s.piperConfig)
	if err != nil || result == nil {
		return nil, err
	}

	return &models.GameReportTypeResp{Key: *result}, nil
}

func (s *Server) GetGameReportInstancesOrByDates(ctx context.Context, r *http.Request) (interface{}, error) {
	userID := betterformat(pat.Param(r, "userid"))
	gameID := betterformat(pat.Param(r, "game"))
	reportType := betterformat(pat.Param(r, "report"))

	startDate := r.URL.Query().Get("start_date")
	endDate := r.URL.Query().Get("end_date")

	// Track get report instance usage
	trackingEvent := models.SpadeEvent{
		UserID:       userID,
		DomainID:     gameID,
		ReportID:     reportType,
		ClientID:     r.Header.Get("Twitch-Client-ID"),
		IsFirstParty: true,
		Action:       "get_game_report_by_date",
		StartDateStr: startDate,
		EndDateStr:   endDate,
		IfSucceed:    false,
	}

	result, err := s.getGameReportByDate(ctx, userID, gameID, reportType, startDate, endDate)
	s.reportTracking(ctx, trackingEvent, trackingGamesBackend, err)
	return result, err
}

func (s *Server) getGameReportByDate(ctx context.Context, userID, gameID, reportName, startDate, endDate string) (interface{}, error) {
	ok, err := s.gamesBackend.UserCanAccess(ctx, "games", userID, gameID, reportName)
	if !ok || err != nil {
		return "", err
	}

	url, err := s.gamesBackend.GetGameReportByDate(ctx, gameID, reportName, startDate, endDate, s.piperConfig)
	if err != nil {
		return "", err
	}

	wrapped := wrapResults("url", url)

	return wrapped, nil
}

func (s *Server) getAllReportDownloadURLs(ctx context.Context, userID, gameID, reportID, startDate, endDate string, limit, offset int) (interface{}, error) {
	result, pagination, err := s.gamesBackend.GetAllReportDownloadURLs(ctx, userID, gameID, reportID, startDate, endDate, limit, offset, s.piperConfig)

	if err != nil || result == nil {
		return &models.HelixReportsResp{Data: []models.HelixReport{}}, err
	}

	return &models.HelixReportsResp{Data: result, Pagination: pagination}, nil
}
