package api

import (
	"net/http"

	"strconv"

	"code.justin.tv/insights/piper-service/models"
	"goji.io/pat"
	"golang.org/x/net/context"
)

func (s *Server) GetGameReportsForHelix(ctx context.Context, r *http.Request) (interface{}, error) {
	userID := betterformat(pat.Param(r, "userid"))
	gameID := r.URL.Query().Get("game_id")
	reportType := r.URL.Query().Get("type")
	startDate := r.URL.Query().Get("start_date")
	endDate := r.URL.Query().Get("end_date")
	var limit, offset int
	strLimit := r.URL.Query().Get("limit")
	strOffset := r.URL.Query().Get("offset")
	var err error

	// Track get game report usage
	trackingEvent := models.SpadeEvent{
		UserID:       userID,
		DomainID:     gameID,
		ReportID:     reportType,
		ClientID:     r.Header.Get("Twitch-Client-ID"),
		IsFirstParty: false,
		Action:       "helix_get_game_reports",
		IfSucceed:    false,
	}

	if strLimit == "" {
		// A good limit should be 100, but since we haven't announced with Helix about this change to external customer
		// So I set to 1000 which should return all results to all games, we could change back to either 50/100 later when officially launching
		limit = 1000
	} else {
		if limit, err = strconv.Atoi(strLimit); err != nil {
			s.reportTracking(ctx, trackingEvent, trackingGamesBackend, err)
			return nil, models.ErrInvalidLimit
		}
	}

	if strOffset == "" {
		offset = 0
	} else {
		if offset, err = strconv.Atoi(strOffset); err != nil {
			s.reportTracking(ctx, trackingEvent, trackingGamesBackend, err)
			return nil, models.ErrInvalidOffset
		}
	}

	// Track get game report usage
	s.gamesBackend.TrackGamesReportUsage(ctx, models.SpadeEvent{
		UserID:       userID,
		DomainID:     gameID,
		ReportID:     reportType,
		ClientID:     r.Header.Get("Twitch-Client-ID"),
		IsFirstParty: false,
		Action:       "helix_get_game_reports",
		StartDateStr: startDate,
		EndDateStr:   endDate,
	}, s.piperConfig)

	reports, err := s.getAllReportDownloadURLs(ctx, userID, gameID, reportType, startDate, endDate, limit, offset)
	s.reportTracking(ctx, trackingEvent, trackingGamesBackend, err)
	return reports, err
}

func (s *Server) GetExtensionReportsForHelix(ctx context.Context, r *http.Request) (interface{}, error) {
	userID := betterformat(pat.Param(r, "userid"))
	extensionID := r.URL.Query().Get("extension_id")
	reportType := r.URL.Query().Get("type")
	startDate := r.URL.Query().Get("start_date")
	endDate := r.URL.Query().Get("end_date")
	var limit, offset int
	strLimit := r.URL.Query().Get("limit")
	strOffset := r.URL.Query().Get("offset")

	var err error
	// Tracking Helix endpoint for extension downloading
	trackingEvent := models.SpadeEvent{
		UserID:       userID,
		DomainID:     extensionID,
		ReportID:     reportType,
		ClientID:     r.Header.Get("Twitch-Client-ID"),
		IsFirstParty: false,
		Action:       "helix_get_extension_reports",
		StartDateStr: startDate,
		EndDateStr:   endDate,
		IfSucceed:    false,
	}

	if extensionID != "" {
		ok, err := s.extensionsBackend.UserCanAccess(ctx, userID, extensionID)
		if !ok || err != nil {
			s.reportTracking(ctx, trackingEvent, trackingExtensionsBackend, err)
			return nil, err
		}
	}

	if strLimit == "" {
		limit = 100
	} else {
		if limit, err = strconv.Atoi(strLimit); err != nil {
			s.reportTracking(ctx, trackingEvent, trackingExtensionsBackend, err)
			return nil, models.ErrInvalidLimit
		}
	}

	if strOffset == "" {
		offset = 0
	} else {
		if offset, err = strconv.Atoi(strOffset); err != nil {
			s.reportTracking(ctx, trackingEvent, trackingExtensionsBackend, err)
			return nil, models.ErrInvalidOffset
		}
	}

	reports, pagination, err := s.extensionsBackend.GetAllReportDownloadURLs(ctx, userID, extensionID, reportType, startDate, endDate, limit, offset, s.piperConfig)
	if err != nil {
		s.reportTracking(ctx, trackingEvent, trackingExtensionsBackend, err)
		return &models.HelixReportsResp{}, err
	}
	s.reportTracking(ctx, trackingEvent, trackingExtensionsBackend, nil)
	return &models.HelixReportsResp{Data: reports, Pagination: pagination}, nil
}
