package api

import (
	"net/http"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"

	"fmt"

	"strconv"
	"time"

	"code.justin.tv/insights/piper-service/models"
	"goji.io/pat"
	"golang.org/x/net/context"
)

func (s *Server) GetOnePagerData(ctx context.Context, r *http.Request) (interface{}, error) {
	userID := betterformat(pat.Param(r, "user_id"))
	gameID := betterformat(pat.Param(r, "game_id"))
	ifNeedUserImg := r.URL.Query().Get("add_user_image")
	quarterStr := r.URL.Query().Get("quarter")
	yearStr := r.URL.Query().Get("year")

	quarter, err := strconv.Atoi(quarterStr)
	if err != nil || quarter < 1 || quarter > 4 {
		return time.Time{}, models.ErrInvalidDateFormat
	}

	_, err = strconv.Atoi(yearStr)
	if err != nil {
		return time.Time{}, models.ErrInvalidDateFormat
	}

	needUserImg, err := strconv.ParseBool(ifNeedUserImg)
	if err != nil {
		needUserImg = false
	}

	company, err := s.getCompanyByUser(ctx, userID)
	if err != nil {
		return nil, err
	}

	isWhitelisted, err := s.onePagerBackend.UserCanAccess(ctx, userID, gameID, company.Id)
	if err != nil {
		return nil, err
	}

	resp, err := s.onePagerBackend.GetGameOnePager(ctx, gameID, yearStr, quarterStr)
	if err != nil {
		return nil, err
	}

	if needUserImg {
		err := s.onePagerBackend.AddProfileImageToResp(ctx, &resp)
		if err != nil {
			return nil, err
		}
	}

	event := "get_games_one_pager"
	if needUserImg {
		event = "get_games_one_pager_poster_view"
	}

	trackingEvent := models.SpadeEvent{
		UserID:       userID,
		DomainID:     gameID,
		CompanyIDStr: company.Id,
		CompanyName:  company.CompanyName,
		DomainName:   resp.GameInfo.GameName,
		IsStaff:      isWhitelisted,
		ReportID:     "quarterly_fact_sheet",
		IfSucceed:    false,
		ClientID:     r.Header.Get("Twitch-Client-ID"),
		IsFirstParty: true,
		StartDateStr: fmt.Sprintf("%s-%02d-01T00:00:00Z", yearStr, time.Month((quarter-1)*3+1)),
		Action:       event,
	}
	s.reportTracking(ctx, trackingEvent, trackingOnePagerBackend, nil)

	return resp, nil
}

func (s *Server) GetQuarterStartRange(ctx context.Context, r *http.Request) (interface{}, error) {
	userID := betterformat(pat.Param(r, "user_id"))
	gameID := betterformat(pat.Param(r, "game_id"))
	isDefaultStr := r.URL.Query().Get("default")

	var quarterStartRange models.QuarterStartRange
	var err error

	isDefault, err := strconv.ParseBool(isDefaultStr)

	company, err := s.getCompanyByUser(ctx, userID)
	if err != nil {
		return nil, err
	}

	isWhitelisted, err := s.onePagerBackend.UserCanAccess(ctx, userID, gameID, company.Id)
	if err != nil {
		return nil, err
	}

	trackingEvent := models.SpadeEvent{
		UserID:       userID,
		DomainID:     gameID,
		CompanyIDStr: company.Id,
		CompanyName:  company.CompanyName,
		IsStaff:      isWhitelisted,
		ReportID:     "quarterly_fact_sheet",
		IfSucceed:    false,
		ClientID:     r.Header.Get("Twitch-Client-ID"),
		IsFirstParty: true,
		Action:       "get_quarter_start_range",
	}
	s.reportTracking(ctx, trackingEvent, trackingOnePagerBackend, nil)

	if isDefault {
		quarterStartRange, err = s.onePagerBackend.GetDefaultQuarterStartRange(ctx)
	} else {
		quarterStartRange, err = s.onePagerBackend.GetQuarterStartRange(ctx, gameID)
	}

	if err != nil {
		return nil, err
	}
	return quarterStartRange, nil
}

func (s *Server) GetOnePagerDownloadURL(ctx context.Context, r *http.Request) (interface{}, error) {
	userID := betterformat(pat.Param(r, "user_id"))
	gameID := betterformat(pat.Param(r, "game_id"))

	quarterStr := r.URL.Query().Get("quarter")
	yearStr := r.URL.Query().Get("year")

	quarter, err := strconv.Atoi(quarterStr)
	if err != nil || quarter < 1 || quarter > 4 {
		return time.Time{}, models.ErrInvalidDateFormat
	}

	_, err = strconv.Atoi(yearStr)
	if err != nil {
		return time.Time{}, models.ErrInvalidDateFormat
	}

	company, err := s.getCompanyByUser(ctx, userID)
	if err != nil {
		return nil, err
	}

	isWhitelisted, err := s.onePagerBackend.UserCanAccess(ctx, userID, gameID, company.Id)
	if err != nil {
		return nil, err
	}

	result, gameName, err := s.onePagerBackend.GetGameOnePagerDownloadURL(ctx, gameID, yearStr, quarterStr, s.piperConfig)

	if err != nil {
		return nil, err
	}

	trackingEvent := models.SpadeEvent{
		UserID:       userID,
		DomainID:     gameID,
		CompanyIDStr: company.Id,
		CompanyName:  company.CompanyName,
		DomainName:   gameName,
		IsStaff:      isWhitelisted,
		ReportID:     "quarterly_fact_sheet",
		IfSucceed:    true,
		ClientID:     r.Header.Get("Twitch-Client-ID"),
		IsFirstParty: true,
		StartDateStr: fmt.Sprintf("%s-%02d-01T00:00:00Z", yearStr, time.Month((quarter-1)*3+1)),
		Action:       "download_games_one_pager_report",
	}
	s.reportTracking(ctx, trackingEvent, trackingOnePagerBackend, nil)

	return result, nil
}

// gets company by user to be used for metric reporting.
// if user is not in a company, a blank company is returned with blank Id and CompanyName.
// this returns an error when RBAC errors.
func (s *Server) getCompanyByUser(ctx context.Context, userID string) (rbacrpc.Company, error) {
	company, err := s.rbacClient.GetCompanyByUserID(ctx, userID)
	if err != nil && err != models.ErrUserNoCompany {
		return company, err
	}
	return company, nil
}
