package api

import (
	"fmt"
	"net"
	"net/http"
	"strings"

	"code.justin.tv/chat/golibs/errx"

	"github.com/aws/aws-sdk-go/aws/awserr"
	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/logx"

	"bytes"
	"encoding/json"
	"strconv"

	"os"

	"code.justin.tv/insights/piper-service/models"
)

const (
	trackingGamesBackend      = "game"
	trackingExtensionsBackend = "extension"
	trackingOnePagerBackend   = "one_pager"
	trackingModsBackend       = "mods"
	trackingDropsBackend      = "drops"
)

func serveJSON(ctx context.Context, status int, w http.ResponseWriter, resp interface{}) {
	jsonStr, err := json.Marshal(resp)
	jsonStr = bytes.Replace(jsonStr, []byte("\\u0026"), []byte("&"), -1)
	if err != nil {
		logServeJSONError(ctx, err)
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	w.Header().Set("Content-Length", strconv.Itoa(len(jsonStr)))
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(status)

	_, err = w.Write(jsonStr)
	if err != nil {
		logServeJSONError(ctx, err)
	}
}

func logServeJSONError(ctx context.Context, err error) {
	logx.Error(ctx, err, logx.Fields{"server_error": fmt.Sprintf("Error serving reponse: %v", err)})
}

func reportError(ctx context.Context, err error) int {
	var status int

	unWrappedErr := errx.Cause(err)
	switch e := unWrappedErr.(type) {
	case awserr.RequestFailure:
		status = e.StatusCode()
	case models.CodedError:
		status = e.StatusCodeValue
	default:
		status = http.StatusInternalServerError
	}

	if status >= 500 {
		logx.Error(ctx, err)
	}
	return status
}

func getIPFromRequest(r *http.Request) string {
	ipStr := fetchLastHeader(r, "X-Forwarded-For")
	if ipStr == "" {
		host, _, err := net.SplitHostPort(r.RemoteAddr)
		if err != nil {
			return ""
		}
		ipStr = host
	}
	ip := net.ParseIP(ipStr)
	if ip == nil {
		return ""
	}
	return ipStr
}

func fetchLastHeader(r *http.Request, name string) string {
	result := r.Header.Get(name)
	if result == "" {
		return ""
	}
	values := strings.Split(result, ",")
	return strings.TrimSpace(values[len(values)-1])
}

func getTwitchRepositoryFromRequest(r *http.Request) string {
	source := r.Header.Get("Twitch-Repository")
	if source == "" {
		return "unknown"
	}
	return source
}

func wrapResults(key string, val interface{}) map[string]interface{} {
	wrapped := make(map[string]interface{})
	wrapped[key] = val
	return wrapped
}

func betterformat(str string) string {
	return strings.ToLower(str)
}

func (s *Server) reportTracking(ctx context.Context, trackingEvent models.SpadeEvent, backendType string, err error) {
	trackingEvent.Environment = environment()

	trackingEvent.IfSucceed = err == nil

	if backendType == "extension" {
		s.extensionsBackend.TrackExtensionReportUsage(ctx, trackingEvent)
	} else if backendType == "game" {
		s.gamesBackend.TrackGamesReportUsage(ctx, trackingEvent, s.piperConfig)
	} else if backendType == "one_pager" {
		s.onePagerBackend.TrackOnePagerUsage(ctx, trackingEvent)
	} else if backendType == "mods" {
		s.modsBackend.TrackModsReportUsage(ctx, trackingEvent)
	} else if backendType == "drops" {
		s.dropsBackend.TrackDropsReportUsage(ctx, trackingEvent)
	}
}

func environment() string {
	environment := os.Getenv("ENVIRONMENT")
	if environment == "" {
		environment = "development"
	}
	return environment
}
