package drops

import (
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"

	"code.justin.tv/insights/piper-service/backend/mocks"
	s3reportmock "code.justin.tv/insights/piper-service/backend/s3report/mocks"
	piperdbmock "code.justin.tv/insights/piper-service/internal/clients/piperdb/mocks"
	rbacmock "code.justin.tv/insights/piper-service/internal/clients/rbac/mocks"
	s3mock "code.justin.tv/insights/piper-service/internal/clients/s3/mocks"
	spademock "code.justin.tv/insights/piper-service/internal/clients/spade/mocks"
	usersmock "code.justin.tv/insights/piper-service/internal/clients/users/mocks"
	piperconfig "code.justin.tv/insights/piper-service/internal/config"

	"code.justin.tv/insights/piper-service/models"
)

const (
	staffUser         = "12345"
	nonStaffUser      = "23456"
	invalidReportType = "kappa"
)

type GetDropReportURLTest struct {
	suite.Suite
	*require.Assertions

	mockS3       *s3mock.Client
	mockSpade    *spademock.Client
	mockPiperDB  *piperdbmock.Client
	mockRbac     *rbacmock.Client
	mockUsers    *usersmock.Client
	mockS3Report *s3reportmock.Backend
	mockCacher   *mocks.Cacher

	backend     Backend
	piperconfig *piperconfig.PiperConfig
	ctx         context.Context

	startDate time.Time
	endDate   time.Time
}

func (suite *GetDropReportURLTest) SetupTest() {
	var err error

	// tests will run `require` instead of `assert` by default
	suite.Assertions = suite.Require()

	suite.mockS3 = &s3mock.Client{}
	suite.mockSpade = &spademock.Client{}
	suite.mockPiperDB = &piperdbmock.Client{}
	suite.mockRbac = &rbacmock.Client{}
	suite.mockUsers = &usersmock.Client{}
	suite.mockS3Report = &s3reportmock.Backend{}
	suite.mockCacher = &mocks.Cacher{}

	suite.backend, err = NewBackend(suite.mockS3, suite.mockSpade, suite.mockPiperDB, suite.mockRbac, suite.mockUsers, suite.mockS3Report, suite.mockCacher)
	suite.piperconfig = &piperconfig.PiperConfig{ReportBucket: "a_bucket", DynamicReportPrefix: "dynamic/"}
	suite.ctx = context.Background()

	suite.startDate = time.Now().AddDate(0, 0, 1).UTC()
	suite.endDate = time.Now().AddDate(0, 0, 5).UTC()

	suite.NoError(err)
}

func (suite *GetDropReportURLTest) TearDownTest() {
	suite.mockS3.AssertExpectations(suite.T())
	suite.mockSpade.AssertExpectations(suite.T())
	suite.mockPiperDB.AssertExpectations(suite.T())
	suite.mockRbac.AssertExpectations(suite.T())
	suite.mockUsers.AssertExpectations(suite.T())
	suite.mockS3Report.AssertExpectations(suite.T())
	suite.mockCacher.AssertExpectations(suite.T())
}

func (suite *GetDropReportURLTest) TestInvalidReportType() {
	url, err := suite.backend.GetDropReportURL(suite.ctx, staffUser, "601", invalidReportType, "", *suite.piperconfig)
	suite.Equal("", url)
	suite.Equal(models.ErrInvalidReportType, err)
}

func (suite *GetDropReportURLTest) TestGetDropsCampaignsOverviewReport() {
	reportType := "campaigns_overview"
	campaignID := "drops-campaign-3000"

	startDateStr := suite.startDate.Format("2006-01-02")
	endDateStr := suite.endDate.Format("2006-01-02")

	suite.mockUsers.On("IsStaff", suite.ctx, staffUser).Return(true, nil)

	suite.mockPiperDB.On("GetDomainReportDateRange", suite.ctx, "drops", reportType, campaignID).Return(suite.startDate, suite.endDate, nil)
	suite.mockPiperDB.On("GetDropsReportByDate", suite.ctx, campaignID, reportType, startDateStr, endDateStr).Return([][]string{}, nil)

	suite.mockS3Report.On("SaveReportToS3", suite.ctx, [][]string{}, campaignID, reportType, startDateStr, endDateStr, suite.piperconfig.ReportBucket, suite.piperconfig.DynamicReportPrefix).Return("drops-campaign-3000_campaigns_overview_2020-07-15_2020-07-19.csv", nil)

	suite.mockS3.On("PreSignURL", suite.piperconfig.ReportBucket, suite.piperconfig.DynamicReportPrefix+"drops-campaign-3000_campaigns_overview_2020-07-15_2020-07-19.csv", "drops-campaign-3000_campaigns_overview_2020-07-15.csv", urlExpireMinutes).Return("Presigned URL", nil)

	url, err := suite.backend.GetDropReportURL(suite.ctx, staffUser, campaignID, reportType, "", *suite.piperconfig)

	suite.NoError(err)
	suite.Equal("Presigned URL", url)
}

func (suite *GetDropReportURLTest) TestGetDropsCampaignsTopStreamersReport() {
	reportType := "campaigns_top_streamers"
	campaignID := "drops-campaign-3"

	startDateStr := suite.startDate.Format("2006-01-02")
	endDateStr := suite.endDate.Format("2006-01-02")

	suite.mockUsers.On("IsStaff", suite.ctx, staffUser).Return(true, nil)

	suite.mockPiperDB.On("GetDomainReportDateRange", suite.ctx, "drops", reportType, campaignID).Return(suite.startDate, suite.endDate, nil)
	suite.mockPiperDB.On("GetDropsReportByDate", suite.ctx, campaignID, reportType, startDateStr, endDateStr).Return([][]string{}, nil)

	suite.mockS3Report.On("SaveReportToS3", suite.ctx, [][]string{}, campaignID, reportType, startDateStr, endDateStr, suite.piperconfig.ReportBucket, suite.piperconfig.DynamicReportPrefix).Return("drops-campaign-3_campaigns_top_streamers_2020-07-15_2020-07-19.csv", nil)

	suite.mockS3.On("PreSignURL", suite.piperconfig.ReportBucket, suite.piperconfig.DynamicReportPrefix+"drops-campaign-3_campaigns_top_streamers_2020-07-15_2020-07-19.csv", "drops-campaign-3_campaigns_top_streamers_2020-07-15.csv", urlExpireMinutes).Return("Presigned URL", nil)

	url, err := suite.backend.GetDropReportURL(suite.ctx, staffUser, campaignID, reportType, "", *suite.piperconfig)

	suite.NoError(err)
	suite.Equal("Presigned URL", url)
}

func TestGetDropReportURLTest(t *testing.T) {
	suite.Run(t, new(GetDropReportURLTest))
}
