package drops

import (
	"fmt"

	"code.justin.tv/insights/piper-service/lib/reports"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

// UserCanAccess validates if a user has access to Drops reports
// by checking whether they have permissions to the corresponding organization
func (c *dropsImpl) UserCanAccess(ctx context.Context, userID, reportType, orgID string) (bool, error) {
	isValidReport := false
	for _, whitelistedReport := range reports.GetWhitelistedReportTypesByDomain("drops") {
		if reportType == whitelistedReport {
			isValidReport = true
		}
	}
	if !isValidReport {
		return false, models.ErrInvalidReportType
	}

	// check if user is staff
	isStaff, err := c.users.IsStaff(ctx, userID)
	if err != nil {
		logx.Warn(ctx, fmt.Sprintf("Couldn't retrieve staff user info: %v", err))
	}
	if isStaff && err == nil {
		return true, nil
	}

	// if user is not staff, check rbac for permissions
	// since rbac doesn't have drops campaigns as resource, we validate with their orgID instead
	err = c.rbac.Validate(ctx, &rbacrpc.ValidateQuery{
		Permission:   "drops::analytics::view",
		ResourceId:   orgID,
		ResourceType: "company",
		UserId:       userID,
	})
	if err != nil {
		logx.Warn(ctx, fmt.Sprintf("RBAC validation failed: %v", err))
		return false, models.ErrAccessForbidden
	}

	return true, nil
}
