package drops

import (
	"testing"

	"code.justin.tv/insights/piper-service/backend/mocks"
	s3reportmock "code.justin.tv/insights/piper-service/backend/s3report/mocks"
	piperdbmock "code.justin.tv/insights/piper-service/internal/clients/piperdb/mocks"
	rbacmock "code.justin.tv/insights/piper-service/internal/clients/rbac/mocks"
	s3mock "code.justin.tv/insights/piper-service/internal/clients/s3/mocks"
	spademock "code.justin.tv/insights/piper-service/internal/clients/spade/mocks"
	usersmock "code.justin.tv/insights/piper-service/internal/clients/users/mocks"
	"code.justin.tv/insights/piper-service/models"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

type UserCanAccessTest struct {
	suite.Suite
	*require.Assertions

	mockS3       *s3mock.Client
	mockSpade    *spademock.Client
	mockPiperDB  *piperdbmock.Client
	mockRbac     *rbacmock.Client
	mockUsers    *usersmock.Client
	mockS3Report *s3reportmock.Backend
	mockCacher   *mocks.Cacher

	backend Backend

	ctx context.Context
}

func (suite *UserCanAccessTest) SetupTest() {
	var err error

	// tests will run `require` instead of `assert` by default
	suite.Assertions = suite.Require()

	suite.mockS3 = &s3mock.Client{}
	suite.mockSpade = &spademock.Client{}
	suite.mockPiperDB = &piperdbmock.Client{}
	suite.mockRbac = &rbacmock.Client{}
	suite.mockUsers = &usersmock.Client{}
	suite.mockS3Report = &s3reportmock.Backend{}
	suite.mockCacher = &mocks.Cacher{}

	suite.backend, err = NewBackend(suite.mockS3, suite.mockSpade, suite.mockPiperDB, suite.mockRbac, suite.mockUsers, suite.mockS3Report, suite.mockCacher)
	suite.ctx = context.Background()

	suite.NoError(err)
}

func (suite *UserCanAccessTest) TearDownTest() {
	suite.mockS3.AssertExpectations(suite.T())
	suite.mockSpade.AssertExpectations(suite.T())
	suite.mockPiperDB.AssertExpectations(suite.T())
	suite.mockRbac.AssertExpectations(suite.T())
	suite.mockUsers.AssertExpectations(suite.T())
	suite.mockS3Report.AssertExpectations(suite.T())
	suite.mockCacher.AssertExpectations(suite.T())
}

func (suite *UserCanAccessTest) TestUserCanAccessStaff() {
	userID := "1234"
	reportType := "campaigns_overview"

	suite.mockUsers.On("IsStaff", suite.ctx, userID).Return(true, nil)

	canAccess, err := suite.backend.UserCanAccess(suite.ctx, userID, reportType, "321321321")
	suite.NoError(err)
	suite.True(canAccess)
}

func (suite *UserCanAccessTest) TestUserCanAccessNonStaff() {
	userID := "1111"
	reportType := "campaigns_overview"

	suite.mockRbac.On("Validate", suite.ctx, mock.Anything).Return(nil)
	suite.mockUsers.On("IsStaff", suite.ctx, userID).Return(false, nil)

	canAccess, err := suite.backend.UserCanAccess(suite.ctx, userID, reportType, "321321321")
	suite.NoError(err)
	suite.True(canAccess)
}

func (suite *UserCanAccessTest) TestUserCanAccessInvalidReportType() {
	userID := "1111"
	invalidReportType := "campaigns_overview_1000"

	canAccess, err := suite.backend.UserCanAccess(suite.ctx, userID, invalidReportType, "321321321")
	suite.Equal(err, models.ErrInvalidReportType)
	suite.False(canAccess)
}

func (suite *UserCanAccessTest) TestUserCanAccessNonStaffNoViewPermission() {
	userID := "1111"
	reportType := "campaigns_overview"

	suite.mockRbac.On("Validate", suite.ctx, mock.Anything).Return(models.ErrAccessForbidden)
	suite.mockUsers.On("IsStaff", suite.ctx, userID).Return(false, nil)

	canAccess, err := suite.backend.UserCanAccess(suite.ctx, userID, reportType, "321321321")
	suite.Equal(err, models.ErrAccessForbidden)
	suite.False(canAccess)
}

func TestUserCanAccessSuite(t *testing.T) {
	suite.Run(t, new(UserCanAccessTest))
}
