package extensions

import (
	"time"

	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/insights/piper-service/backend"
	"code.justin.tv/insights/piper-service/backend/s3report"
	"code.justin.tv/insights/piper-service/internal/clients/owl"
	"code.justin.tv/insights/piper-service/internal/clients/piperdb"
	"code.justin.tv/insights/piper-service/internal/clients/rbac"
	"code.justin.tv/insights/piper-service/internal/clients/s3"
	"code.justin.tv/insights/piper-service/internal/clients/users"
	cfg "code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

//go:generate mockery -name Backend
type Backend interface {
	GetReportTypes(ctx context.Context, userID, extensionID string, config cfg.PiperConfig) (*models.ReportType, error)
	GetReportByDate(ctx context.Context, extensionID, reportType, startDate, endDate string, config cfg.PiperConfig) (string, error)
	GetAllReportDownloadURLs(ctx context.Context, userID, extensionID, reportType, startDate, endDate string, limit, offset int, config cfg.PiperConfig) (models.HelixReports, *models.Pagination, error)
	UserCanAccess(ctx context.Context, userID, extensionID string) (bool, error)
	TrackExtensionReportUsage(ctx context.Context, extensionsEvent models.SpadeEvent)
}

type extensionsRepoImpl struct {
	s3            s3.Client
	cacher        backend.Cacher
	spade         spade.Client
	users         users.Client
	owl           owl.Client
	s3report      s3report.Backend
	piperdbclient piperdb.Client
	rbac          rbac.Client
}

func NewBackend(s3client s3.Client, cacher backend.Cacher, spadeclient spade.Client, usersclient users.Client, owlclient owl.Client, s3report s3report.Backend, piperdbclient piperdb.Client, rbacclient rbac.Client) (*extensionsRepoImpl, error) {
	b := &extensionsRepoImpl{
		s3:            s3client,
		cacher:        cacher,
		spade:         spadeclient,
		users:         usersclient,
		owl:           owlclient,
		s3report:      s3report,
		piperdbclient: piperdbclient,
		rbac:          rbacclient,
	}

	return b, nil
}

const (
	backendDomain    = "extensions"     // backendDomain is a Enum. its values can be "games", "extensions", "mods" and "one_pager". it's used in tracking tables and piper database tables.
	asyncTimeout     = 20 * time.Second // AsyncTimeout is the time we want to wait for async work to complete
	urlExpireMinutes = 5
	keyDelim         = "|"
)
