package extensions

import (
	"sort"
	"strings"

	"fmt"

	"sync"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	cfg "code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/lib/reports"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

type goRoutineReport struct {
	report models.HelixReport
	err    error
}

func (c *extensionsRepoImpl) GetAllReportDownloadURLs(ctx context.Context, userID, extensionID, reportType, startDate, endDate string, limit, offset int, config cfg.PiperConfig) (models.HelixReports, *models.Pagination, error) {
	err := reports.IsValidLimit(limit)
	if err != nil {
		return nil, nil, err
	}

	// early check only one of the params is set, bad time format, and start is after end
	err = reports.IsValidDates(startDate, endDate)
	if err != nil {
		return nil, nil, err
	}

	// early check invalid report type
	err = reports.IsValidReportType(backendDomain, reportType)
	if err != nil {
		return nil, nil, err
	}

	// early check if the given extension id is valid
	err = c.owl.IsValidExtension(ctx, extensionID)
	if err != nil {
		return nil, nil, err
	}

	// get extensions that this user can access
	extensionIDs, err := c.getValidUserExtensions(ctx, userID, extensionID)
	if err != nil {
		return nil, nil, err
	}

	// set up domain, report, start date, end date with valid value
	domainReportTypes := reports.GetValidReportTypes(backendDomain, reportType) // either reportType or all the types if reportType == ""
	var fullExtensionReports models.HelixReports
	for _, r := range domainReportTypes {
		for _, g := range extensionIDs {
			fullExtensionReports = append(fullExtensionReports, models.HelixReport{
				ID:        g,
				Type:      r,
				StartDate: startDate,
				EndDate:   endDate,
			})
		}
	}

	if len(fullExtensionReports) == 0 {
		return nil, nil, errx.New(models.ErrReportNotFound)
	}

	if offset >= len(fullExtensionReports) {
		return nil, nil, models.ErrInvalidOffset
	}

	availableHelixReports, err := c.fetchAvailableHelixReports(ctx, userID, fullExtensionReports, config)

	if len(availableHelixReports) == 0 {
		if errx.Cause(err) == models.ErrInvalidDateRange || errx.Cause(err) == models.ErrReportNotFound {
			return models.HelixReports{}, nil, errx.New(err)
		}
		return nil, nil, errx.New(err)
	}

	var extensionReports models.HelixReports
	for _, helixReport := range availableHelixReports {
		extensionReports = append(extensionReports, helixReport)
	}

	sort.Sort(extensionReports)

	pagination := &models.Pagination{Limit: limit, Offset: offset + limit}
	to := offset + limit
	if to >= len(extensionReports) {
		to = len(extensionReports)
		pagination = nil
	}

	extensionReports = extensionReports[offset:to]

	if offset >= len(extensionReports) {
		return nil, nil, models.ErrInvalidOffset
	}

	helixReports, err := c.fetchHelixReportsURL(ctx, userID, extensionReports, config)

	if len(helixReports) == 0 || err != nil {
		return models.HelixReports{}, nil, errx.New(err)
	}

	return helixReports, pagination, nil
}

func (c *extensionsRepoImpl) getReportDefaultDates(ctx context.Context, userID string, report models.HelixReport, config cfg.PiperConfig) (models.HelixReport, error) {
	key := userID + keyDelim + report.ID + keyDelim + report.Type + keyDelim + report.StartDate + keyDelim + report.EndDate

	prop, found := c.cacher.GetStringProperties(ctx, key)
	if found {
		defaultDates := strings.Split(prop, keyDelim)
		if len(defaultDates) == 2 {
			report.StartDate = defaultDates[0]
			report.EndDate = defaultDates[1]
			return report, nil
		}
	}
	reportStartDate, reportEndDate, err := c.GetFittedDateRange(ctx, report.ID, report.Type, report.StartDate, report.EndDate)
	if err != nil || reportStartDate == "" || reportEndDate == "" {
		return report, err
	}

	report.StartDate = reportStartDate
	report.EndDate = reportEndDate

	prop = reportStartDate + keyDelim + reportEndDate

	err = c.cacher.CacheStringProperties(ctx, key, prop)
	if err != nil {
		logx.Error(ctx, fmt.Sprintf("failed to call get helix report dates for %s in cache", key))
	}
	return report, nil
}

func recoverFetchReport(ctx context.Context, domainID string) {
	if r := recover(); r != nil {
		logx.Error(ctx, fmt.Sprintf("panic in async job to fetch Helix report for %s", domainID))
	}
}

func (c *extensionsRepoImpl) fetchAvailableHelixReports(ctx context.Context, userID string, fullExtensionReports []models.HelixReport, config cfg.PiperConfig) (models.HelixReports, error) {
	var availableHelixReports []models.HelixReport

	var wg sync.WaitGroup
	resCh := make(chan *goRoutineReport, len(fullExtensionReports))

	for _, extensionReport := range fullExtensionReports {
		bCtx, cancel := context.WithTimeout(context.Background(), asyncTimeout)

		wg.Add(1)

		go func(ctx context.Context, report models.HelixReport) {
			// defer recover to avoid a panic in a go routine to kill the service
			defer cancel()
			defer recoverFetchReport(ctx, report.ID)

			result, err := c.getReportDefaultDates(ctx, userID, report, config)
			resCh <- &goRoutineReport{
				result,
				err}

			wg.Done()
			return
		}(bCtx, extensionReport)
	}

	wg.Wait()
	close(resCh)

	var fErr error
	for processedReport := range resCh {
		pErr := processedReport.err
		if pErr == nil && processedReport.report.StartDate != "" && processedReport.report.EndDate != "" {
			availableHelixReports = append(availableHelixReports, processedReport.report)
		} else if pErr != nil && errx.Cause(pErr) != models.ErrReportNotFound && errx.Cause(pErr) != models.ErrInvalidDateRange {
			// if a user queries multiple reports with the same date range, there may be reports hit dates out of range error
			// there may be no reports for a given extension id and report type
			// we do not log these errors
			logx.Error(ctx, fmt.Sprintf("error getReportDefaultDates for user: %s report: %s. %v", userID, processedReport.report, pErr))
		} else {
			// if an error is due to out of bound dates, or no report exists for a extension id and report type combo, we return any of this error for future check
			fErr = pErr
		}
	}

	return availableHelixReports, fErr
}

func (c *extensionsRepoImpl) fetchHelixReportsURL(ctx context.Context, userID string, extensionReports []models.HelixReport, config cfg.PiperConfig) (models.HelixReports, error) {
	var helixReports []models.HelixReport

	var wg sync.WaitGroup
	resCh := make(chan *goRoutineReport, len(extensionReports))

	for _, extensionReport := range extensionReports {
		bCtx, cancel := context.WithTimeout(context.Background(), asyncTimeout)

		wg.Add(1)

		go func(ctx context.Context, report models.HelixReport) {
			// defer recover to avoid a panic in a go routine to kill the service
			defer cancel()
			defer recoverFetchReport(ctx, report.ID)

			url, err := c.generateReportURL(ctx, userID, report.ID, report.Type, report.StartDate, report.EndDate, config)
			report.URL = url

			resCh <- &goRoutineReport{
				report,
				err,
			}
			wg.Done()
		}(bCtx, extensionReport)
	}

	wg.Wait()
	close(resCh)

	var fErr error
	for processedReport := range resCh {
		pErr := processedReport.err
		if pErr == nil && processedReport.report.URL != "" {
			helixReports = append(helixReports, processedReport.report)
		} else if pErr != nil {
			fErr = pErr
			logx.Error(ctx, fmt.Sprintf("error fetching URL for user: %s report: %s. %v", userID, processedReport.report, pErr))
		}
	}

	return helixReports, fErr
}
