package extensions

import (
	"testing"
	"time"

	"code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"code.justin.tv/chat/golibs/errx"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

var (
	shortDate  = "2006-01-02"
	overviewV1 = "overview_v1"
)

type GetAllURLsTest struct {
	extensionSuite
}

func (suite *GetAllURLsTest) TestGetAllURLs_NoExtensionIDNoReportTypeNoDates() {
	cfg := config.PiperConfig{
		ReportBucket:        "a_bucket",
		DynamicReportPrefix: "dynamic/",
	}

	startDate := time.Now().UTC().AddDate(0, 0, -80).Format(shortDate)
	endDate := time.Now().UTC().AddDate(0, 0, -2).Format(shortDate)

	suite.mockOwl.On("GetExtensionsByUserID", mock.Anything, userID).Return([]string{"1212", "1213"}, nil)

	suite.mockOwl.On("GetExtensionName", mock.Anything, "1212").Return("Extension 1")
	suite.mockOwl.On("GetExtensionName", mock.Anything, "1213").Return("Extension 2")

	suite.mockOwl.On("IsValidExtension", mock.Anything, "").Return(nil)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+keyDelim+"1212"+keyDelim+overviewV1+"||").Return(startDate+keyDelim+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+keyDelim+"1213"+keyDelim+overviewV1+"||").Return(startDate+keyDelim+endDate, true)

	suite.mockPiperDB.On("GetExtensionReportByDate", mock.Anything, "1212", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetExtensionReportByDate", mock.Anything, "1213", overviewV1, startDate, endDate).Return([][]string{}, nil)

	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Extension 1", overviewV1, startDate, endDate, cfg.ReportBucket, cfg.DynamicReportPrefix).Return("Extension 1_overview_2018-03-01_2018-04-01.csv", nil)
	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Extension 2", overviewV1, startDate, endDate, cfg.ReportBucket, cfg.DynamicReportPrefix).Return("Extension 2_overview_2018-03-01_2018-04-01.csv", nil)

	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Extension 1_overview_2018-03-01_2018-04-01.csv", "Extension 1_overview_2018-03-01.csv", urlExpireMinutes).Return("URL 1", nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Extension 2_overview_2018-03-01_2018-04-01.csv", "Extension 2_overview_2018-03-01.csv", urlExpireMinutes).Return("URL 2", nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", overviewV1, "", "", 10, 0, cfg)
	assert.NoError(suite.T(), err)
	assert.Nil(suite.T(), pagination)
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 2)

	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1212",
		Type:      overviewV1,
		URL:       "URL 1",
		StartDate: startDate,
		EndDate:   endDate,
	})
	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1213",
		Type:      overviewV1,
		URL:       "URL 2",
		StartDate: startDate,
		EndDate:   endDate,
	})
}

func (suite *GetAllURLsTest) TestGetAllURLs_WithExtensionIDNoReportTypesNoDates() {
	cfg := config.PiperConfig{
		ReportBucket:        "a_bucket",
		DynamicReportPrefix: "dynamic/",
	}

	startDate := time.Now().UTC().AddDate(0, 0, -80).Format(shortDate)
	endDate := time.Now().UTC().AddDate(0, 0, -2).Format(shortDate)

	suite.mockOwl.On("IsValidExtension", mock.Anything, "1212").Return(nil)
	suite.mockOwl.On("GetExtensionName", mock.Anything, "1212").Return("Extension 1")

	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+keyDelim+"1212"+keyDelim+overviewV1+"||").Return(startDate+keyDelim+endDate, true)

	suite.mockPiperDB.On("GetExtensionReportByDate", mock.Anything, "1212", overviewV1, startDate, endDate).Return([][]string{}, nil)

	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Extension 1", overviewV1, startDate, endDate, cfg.ReportBucket, cfg.DynamicReportPrefix).Return("Extension 1_overview_2018-03-01_2018-04-01.csv", nil)

	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Extension 1_overview_2018-03-01_2018-04-01.csv", "Extension 1_overview_2018-03-01.csv", urlExpireMinutes).Return("URL 1", nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "1212", overviewV1, "", "", 10, 0, cfg)
	assert.NoError(suite.T(), err)
	assert.Nil(suite.T(), pagination)
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 1)

	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1212",
		Type:      overviewV1,
		URL:       "URL 1",
		StartDate: startDate,
		EndDate:   endDate,
	})
}

func (suite *GetAllURLsTest) TestGetAllURLs_WithExtensionIDWithReportTypesNoDates() {
	cfg := config.PiperConfig{
		ReportBucket:        "a_bucket",
		DynamicReportPrefix: "dynamic/",
	}

	startDate := time.Now().UTC().AddDate(0, 0, -80).Format(shortDate)
	endDate := time.Now().UTC().AddDate(0, 0, -2).Format(shortDate)

	suite.mockOwl.On("IsValidExtension", mock.Anything, "1212").Return(nil)
	suite.mockOwl.On("GetExtensionName", mock.Anything, "1212").Return("Extension 1")

	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+keyDelim+"1212"+keyDelim+overviewV1+"||").Return(startDate+keyDelim+endDate, true)

	suite.mockPiperDB.On("GetExtensionReportByDate", mock.Anything, "1212", overviewV1, startDate, endDate).Return([][]string{}, nil)

	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Extension 1", overviewV1, startDate, endDate, cfg.ReportBucket, cfg.DynamicReportPrefix).Return("Extension 1_overview_2018-03-01_2018-04-01.csv", nil)

	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Extension 1_overview_2018-03-01_2018-04-01.csv", "Extension 1_overview_2018-03-01.csv", urlExpireMinutes).Return("URL 1", nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "1212", overviewV1, "", "", 10, 0, cfg)
	assert.NoError(suite.T(), err)
	assert.Nil(suite.T(), pagination)
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 1)

	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1212",
		Type:      overviewV1,
		URL:       "URL 1",
		StartDate: startDate,
		EndDate:   endDate,
	})
}

func (suite *GetAllURLsTest) TestGetAllURLs_NoExtensionIDNoReportTypesWithDates() {
	cfg := config.PiperConfig{
		ReportBucket:        "a_bucket",
		DynamicReportPrefix: "dynamic/",
	}

	d := time.Date(2018, 7, 22, 0, 0, 0, 0, time.UTC)
	inputStartDate := d.AddDate(0, 0, -70).Format(time.RFC3339)
	inputEndDate := d.AddDate(0, 0, -10).Format(time.RFC3339)

	startDate := d.AddDate(0, 0, -70).Format(shortDate)
	endDate := d.AddDate(0, 0, -10).Format(shortDate)

	suite.mockOwl.On("GetExtensionsByUserID", mock.Anything, userID).Return([]string{"1212", "1213"}, nil)

	suite.mockOwl.On("GetExtensionName", mock.Anything, "1212").Return("Extension 1")
	suite.mockOwl.On("GetExtensionName", mock.Anything, "1213").Return("Extension 2")

	suite.mockOwl.On("IsValidExtension", mock.Anything, "").Return(nil)

	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+keyDelim+"1212"+keyDelim+overviewV1+keyDelim+inputStartDate+keyDelim+inputEndDate).Return(startDate+keyDelim+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+keyDelim+"1213"+keyDelim+overviewV1+keyDelim+inputStartDate+keyDelim+inputEndDate).Return(startDate+keyDelim+endDate, true)

	suite.mockPiperDB.On("GetExtensionReportByDate", mock.Anything, "1212", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetExtensionReportByDate", mock.Anything, "1213", overviewV1, startDate, endDate).Return([][]string{}, nil)

	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Extension 1", overviewV1, startDate, endDate, cfg.ReportBucket, cfg.DynamicReportPrefix).Return("Extension 1_overview_2018-03-01_2018-04-01.csv", nil)
	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Extension 2", overviewV1, startDate, endDate, cfg.ReportBucket, cfg.DynamicReportPrefix).Return("Extension 2_overview_2018-03-01_2018-04-01.csv", nil)

	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Extension 1_overview_2018-03-01_2018-04-01.csv", "Extension 1_overview_2018-03-01.csv", urlExpireMinutes).Return("URL 1", nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Extension 2_overview_2018-03-01_2018-04-01.csv", "Extension 2_overview_2018-03-01.csv", urlExpireMinutes).Return("URL 2", nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", overviewV1, inputStartDate, inputEndDate, 10, 0, cfg)
	assert.NoError(suite.T(), err)
	assert.Nil(suite.T(), pagination)
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 2)

	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1212",
		Type:      overviewV1,
		URL:       "URL 1",
		StartDate: startDate,
		EndDate:   endDate,
	})
	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1213",
		Type:      overviewV1,
		URL:       "URL 2",
		StartDate: startDate,
		EndDate:   endDate,
	})
}

func (suite *GetAllURLsTest) TestGetAllURLs_Paginated_PageOne() {
	cfg := config.PiperConfig{
		ReportBucket:        "a_bucket",
		DynamicReportPrefix: "dynamic/",
	}

	startDate := time.Now().UTC().AddDate(0, 0, -80).Format(shortDate)
	endDate := time.Now().UTC().AddDate(0, 0, -2).Format(shortDate)

	suite.mockOwl.On("GetExtensionsByUserID", mock.Anything, userID).Return([]string{"1212", "1213", "1214"}, nil)

	suite.mockOwl.On("IsValidExtension", mock.Anything, "").Return(nil)

	suite.mockOwl.On("GetExtensionName", mock.Anything, "1212").Return("Extension 1")
	suite.mockOwl.On("GetExtensionName", mock.Anything, "1213").Return("Extension 2")

	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+keyDelim+"1212"+keyDelim+overviewV1+"||").Return(startDate+keyDelim+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+keyDelim+"1213"+keyDelim+overviewV1+"||").Return(startDate+keyDelim+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+keyDelim+"1214"+keyDelim+overviewV1+"||").Return(startDate+keyDelim+endDate, true)

	suite.mockPiperDB.On("GetExtensionReportByDate", mock.Anything, "1212", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetExtensionReportByDate", mock.Anything, "1213", overviewV1, startDate, endDate).Return([][]string{}, nil)

	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Extension 1", overviewV1, startDate, endDate, cfg.ReportBucket, cfg.DynamicReportPrefix).Return("Extension 1_overview_2018-03-01_2018-04-01.csv", nil)
	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Extension 2", overviewV1, startDate, endDate, cfg.ReportBucket, cfg.DynamicReportPrefix).Return("Extension 2_overview_2018-03-01_2018-04-01.csv", nil)

	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Extension 1_overview_2018-03-01_2018-04-01.csv", "Extension 1_overview_2018-03-01.csv", urlExpireMinutes).Return("URL 1", nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Extension 2_overview_2018-03-01_2018-04-01.csv", "Extension 2_overview_2018-03-01.csv", urlExpireMinutes).Return("URL 2", nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", overviewV1, "", "", 2, 0, cfg)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), pagination.Offset, 2)
	assert.Equal(suite.T(), pagination.Limit, 2)
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 2)

	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1212",
		Type:      overviewV1,
		URL:       "URL 1",
		StartDate: startDate,
		EndDate:   endDate,
	})
	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1213",
		Type:      overviewV1,
		URL:       "URL 2",
		StartDate: startDate,
		EndDate:   endDate,
	})
}

func (suite *GetAllURLsTest) TestGetAllURLs_Paginated_PageTwo() {
	cfg := config.PiperConfig{
		ReportBucket:        "a_bucket",
		DynamicReportPrefix: "dynamic/",
	}

	startDate := time.Now().UTC().AddDate(0, 0, -80).Format(shortDate)
	endDate := time.Now().UTC().AddDate(0, 0, -2).Format(shortDate)

	suite.mockOwl.On("GetExtensionsByUserID", mock.Anything, userID).Return([]string{"1212", "1213", "1214"}, nil)

	suite.mockOwl.On("IsValidExtension", mock.Anything, "").Return(nil)

	suite.mockOwl.On("GetExtensionName", mock.Anything, "1213").Return("Extension 2")
	suite.mockOwl.On("GetExtensionName", mock.Anything, "1214").Return("Extension 3")

	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+keyDelim+"1212"+keyDelim+overviewV1+"||").Return(startDate+keyDelim+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+keyDelim+"1213"+keyDelim+overviewV1+"||").Return(startDate+keyDelim+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+keyDelim+"1214"+keyDelim+overviewV1+"||").Return(startDate+keyDelim+endDate, true)

	suite.mockPiperDB.On("GetExtensionReportByDate", mock.Anything, "1213", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetExtensionReportByDate", mock.Anything, "1214", overviewV1, startDate, endDate).Return([][]string{}, nil)

	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Extension 2", overviewV1, startDate, endDate, cfg.ReportBucket, cfg.DynamicReportPrefix).Return("Extension 2_overview_2018-03-01_2018-04-01.csv", nil)
	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Extension 3", overviewV1, startDate, endDate, cfg.ReportBucket, cfg.DynamicReportPrefix).Return("Extension 3_overview_2018-03-01_2018-04-01.csv", nil)

	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Extension 2_overview_2018-03-01_2018-04-01.csv", "Extension 2_overview_2018-03-01.csv", urlExpireMinutes).Return("URL 2", nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Extension 3_overview_2018-03-01_2018-04-01.csv", "Extension 3_overview_2018-03-01.csv", urlExpireMinutes).Return("URL 3", nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", overviewV1, "", "", 2, 1, cfg)
	assert.NoError(suite.T(), err)
	assert.Nil(suite.T(), pagination)
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 2)

	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1214",
		Type:      overviewV1,
		URL:       "URL 3",
		StartDate: startDate,
		EndDate:   endDate,
	})
	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1213",
		Type:      overviewV1,
		URL:       "URL 2",
		StartDate: startDate,
		EndDate:   endDate,
	})
}

func (suite *GetAllURLsTest) TestGetAllURLs_BadOffset() {
	cfg := config.PiperConfig{
		ReportBucket:        "a_bucket",
		DynamicReportPrefix: "dynamic/",
	}

	suite.mockOwl.On("GetExtensionsByUserID", mock.Anything, userID).Return([]string{"1212", "1213", "1214"}, nil)

	suite.mockOwl.On("IsValidExtension", mock.Anything, "").Return(nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", overviewV1, "", "", 1, 4, cfg)
	assert.Equal(suite.T(), err, models.ErrInvalidOffset)
	assert.Nil(suite.T(), reports)
	assert.Nil(suite.T(), pagination)
}

func (suite *GetAllURLsTest) TestGetAllURLs_BadLimit() {
	cfg := config.PiperConfig{
		ReportBucket:        "a_bucket",
		DynamicReportPrefix: "dynamic/",
	}

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", overviewV1, "", "", 0, 4, cfg)
	assert.Equal(suite.T(), err, models.ErrInvalidLimit)
	assert.Nil(suite.T(), reports)
	assert.Nil(suite.T(), pagination)
}

func (suite *GetAllURLsTest) TestGetAllURLs_BadTimeFormat() {
	var startDate string
	var endDate string
	var err error

	// dates not allowed, it needs to be a RFC3339 timestamp
	startDate = "2018-07-22"
	endDate = "2018-07-24"
	_, _, err = suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", overviewV1, startDate, endDate, 10, 0, config.PiperConfig{})
	suite.EqualError(err, models.ErrInvalidDateFormat.Error())

	// start date needs to be before end date
	startDate = "2018-07-24T00:00:00Z"
	endDate = "2018-07-23T00:00:00Z"
	_, _, err = suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", overviewV1, startDate, endDate, 10, 0, config.PiperConfig{})
	suite.EqualError(err, models.ErrInvalidDateRange.Error())

	// need both start and end date
	startDate = "2018-07-24T00:00:00Z"
	endDate = ""
	_, _, err = suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", overviewV1, startDate, endDate, 10, 0, config.PiperConfig{})
	suite.EqualError(err, models.ErrInvalidDateFormat.Error())
}

func (suite *GetAllURLsTest) TestGetAllURLs_InvalidReportType() {
	cfg := config.PiperConfig{
		ReportBucket:        "a_bucket",
		DynamicReportPrefix: "dynamic/",
	}

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", "notype", "", "", 10, 0, cfg)
	assert.Equal(suite.T(), errx.New(models.ErrInvalidReportType).Error(), err.Error())
	assert.Nil(suite.T(), reports)
	assert.Nil(suite.T(), pagination)
}

func TestGetAllURLsSuite(t *testing.T) {
	suite.Run(t, new(GetAllURLsTest))
}
