package extensions

import (
	"testing"

	"code.justin.tv/insights/piper-service/lib/reports"
	"code.justin.tv/insights/piper-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"

	"code.justin.tv/insights/piper-service/backend/mocks"
	s3reportmock "code.justin.tv/insights/piper-service/backend/s3report/mocks"
	owlmock "code.justin.tv/insights/piper-service/internal/clients/owl/mocks"
	piperdbmock "code.justin.tv/insights/piper-service/internal/clients/piperdb/mocks"
	rbacmock "code.justin.tv/insights/piper-service/internal/clients/rbac/mocks"
	s3mock "code.justin.tv/insights/piper-service/internal/clients/s3/mocks"
	spademock "code.justin.tv/insights/piper-service/internal/clients/spade/mocks"
	usermock "code.justin.tv/insights/piper-service/internal/clients/users/mocks"

	"code.justin.tv/insights/piper-service/internal/config"
)

type extensionSuite struct {
	suite.Suite
	mockS3       *s3mock.Client
	mockCacher   *mocks.Cacher
	mockS3Report *s3reportmock.Backend
	mockSpade    *spademock.Client
	mockUsers    *usermock.Client
	mockOwl      *owlmock.Client
	mockPiperDB  *piperdbmock.Client
	mockRBAC     *rbacmock.Client
	backend      Backend
	cfg          config.PiperConfig
	ctx          context.Context
}

func (suite *extensionSuite) SetupTest() {
	suite.mockS3 = &s3mock.Client{}
	suite.mockSpade = &spademock.Client{}
	suite.mockUsers = &usermock.Client{}
	suite.mockS3Report = &s3reportmock.Backend{}
	suite.mockPiperDB = &piperdbmock.Client{}
	suite.mockCacher = &mocks.Cacher{}
	suite.mockOwl = &owlmock.Client{}
	suite.mockRBAC = &rbacmock.Client{}

	var err error
	suite.cfg = config.PiperConfig{
		ReportBucket:          "a_bucket",
		ExtensionReportPrefix: "extensions/",
	}

	suite.backend, err = NewBackend(suite.mockS3, suite.mockCacher, suite.mockSpade, suite.mockUsers, suite.mockOwl, suite.mockS3Report, suite.mockPiperDB, suite.mockRBAC)
	suite.ctx = context.Background()
	assert.NoError(suite.T(), err)
}

func (suite *extensionSuite) TearDownTest() {
	suite.mockS3.AssertExpectations(suite.T())
	suite.mockSpade.AssertExpectations(suite.T())
	suite.mockUsers.AssertExpectations(suite.T())
	suite.mockS3Report.AssertExpectations(suite.T())
	suite.mockPiperDB.AssertExpectations(suite.T())
	suite.mockCacher.AssertExpectations(suite.T())
	suite.mockOwl.AssertExpectations(suite.T())
	suite.mockRBAC.AssertExpectations(suite.T())
}

type GetReportTypesTest struct {
	extensionSuite
}

const (
	userID        = "12331"
	extensionID   = "kr7m1hz7esldlrptruk0fd76l6yq36"
	extensionName = "WoW Armory"
)

func (suite *GetReportTypesTest) TestGetReportTypeHasReport() {
	reportType := models.ReportType{ID: extensionID, ReportTypes: []string{reports.DevsiteExtensionsReportType}}

	suite.mockPiperDB.On("GetAvailableDomainsWithReports", suite.ctx, backendDomain, reports.DevsiteGamesReportType, []string{extensionID}).Return([]string{extensionID}, nil)
	result, err := suite.backend.GetReportTypes(suite.ctx, userID, extensionID, suite.cfg)
	assert.Nil(suite.T(), err)
	require.NotNil(suite.T(), result)
	assert.Equal(suite.T(), &reportType, result)

}

func (suite *GetReportTypesTest) TestGetReportTypeNoReport() {
	suite.mockPiperDB.On("GetAvailableDomainsWithReports", suite.ctx, backendDomain, reports.DevsiteGamesReportType, []string{extensionID}).Return([]string{}, nil)
	result, err := suite.backend.GetReportTypes(suite.ctx, userID, extensionID, suite.cfg)
	assert.Equal(suite.T(), models.ErrResourceNotFound, err)
	assert.Equal(suite.T(), (*models.ReportType)(nil), result)
}

func TestRunGetReportTypesSuite(t *testing.T) {
	suite.Run(t, new(GetReportTypesTest))
}
