package extensions

import (
	"fmt"

	"code.justin.tv/devrel/devsite-rbac/models/permissions"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/insights/piper-service/internal/clients/owl"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

func (c *extensionsRepoImpl) UserCanAccess(ctx context.Context, userID, extensionID string) (bool, error) {
	isStaff, err := c.users.IsStaff(ctx, userID)
	if err != nil {
		logx.Warn(ctx, fmt.Sprintf("Couldn't retrieve user staff info: %v", err))
	}
	if isStaff {
		return true, nil
	}

	rbacValidateErr := c.rbac.Validate(ctx, &rbacrpc.ValidateQuery{
		UserId:       userID,
		ResourceType: permissions.Extension,
		Permission:   permissions.ViewExtensionInsights,
		ResourceId:   extensionID,
	})
	if rbacValidateErr == nil {
		return true, nil // all good, the user can access this extension in this org
	}
	if rbacValidateErr == models.ErrAccessForbidden {
		logx.Warn(ctx, fmt.Sprintf("RBAC validation failed: %v", rbacValidateErr))
		return false, rbacValidateErr // the extension is in another org, or the user has no permissions to access
	}

	// The error may still be ErrResourceNotFound or ErrInternalError
	// Try Owl to see if the extension owner_id is the same user for personal (org-less) extensions
	isClientIDMatchingUserID, err := c.owl.IsClientIDUserMatching(ctx, userID, extensionID)
	if err == owl.ErrForbiddenClientID {
		logx.Warn(ctx, fmt.Sprintf("Owl validation failed: %v", err))
		return false, models.ErrAccessForbidden
	}
	if err == owl.ErrClientOwnedByOrg {
		// if extension is org owned and owl is catching it and not rbac,
		// there was an issue with rbac. either a 5xx or a bad request.
		// check rollbar.
		return false, errx.New(models.ErrInternalError)
	}
	if err != nil {
		logx.Warn(ctx, fmt.Sprintf("Couldn't retrieve user client info: %v", err))
	}

	if !isClientIDMatchingUserID {
		logx.Warn(ctx, "!isClientIDMatchingUserID")
		return false, models.ErrAccessForbidden
	}

	return true, nil
}
