package extensions

import (
	"testing"

	"code.justin.tv/insights/piper-service/internal/clients/owl"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/insights/piper-service/internal/clients/rbac"
	"code.justin.tv/insights/piper-service/models"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type UserCanAccessTest struct {
	extensionSuite
}

func (suite *UserCanAccessTest) TestUserCanAccessNonStaffCase() {
	suite.mockOwl.On("IsClientIDUserMatching", mock.Anything, userID, extensionID).Return(true, nil)
	suite.mockRBAC.On("Validate", mock.Anything, mock.Anything).Return(rbac.ErrUnknownResource)
	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(false, nil)

	hasAccess, err := suite.backend.UserCanAccess(suite.ctx, userID, extensionID)
	assert.NoError(suite.T(), err)
	assert.True(suite.T(), hasAccess)
}

func (suite *UserCanAccessTest) TestUserCanAccessStaffCase() {
	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(true, nil)

	hasAccess, err := suite.backend.UserCanAccess(suite.ctx, userID, extensionID)
	assert.NoError(suite.T(), err)
	assert.True(suite.T(), hasAccess)
}

func (suite *UserCanAccessTest) TestUserCanAccessUsersServiceError() {
	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(false, errors.New("users service failure"))
	suite.mockRBAC.On("Validate", mock.Anything, mock.Anything).Return(rbac.ErrUnknownResource)
	suite.mockOwl.On("IsClientIDUserMatching", mock.Anything, userID, extensionID).Return(false, nil)

	hasAccess, err := suite.backend.UserCanAccess(suite.ctx, userID, extensionID)
	assert.Equal(suite.T(), err, models.ErrAccessForbidden)
	assert.False(suite.T(), hasAccess)
}

func (suite *UserCanAccessTest) TestUserCanAccessOwlError() {
	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(false, nil)
	suite.mockRBAC.On("Validate", mock.Anything, mock.Anything).Return(rbac.ErrUnknownResource)
	suite.mockOwl.On("IsClientIDUserMatching", mock.Anything, userID, extensionID).Return(false, errors.New("owl failure"))

	hasAccess, err := suite.backend.UserCanAccess(suite.ctx, userID, extensionID)
	assert.Equal(suite.T(), err, models.ErrAccessForbidden)
	assert.False(suite.T(), hasAccess)
}

func (suite *UserCanAccessTest) TestUserCanAccessRBACSuccess() {
	suite.mockRBAC.On("Validate", mock.Anything, mock.Anything).Return(nil)
	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(false, nil)

	hasAccess, err := suite.backend.UserCanAccess(suite.ctx, userID, extensionID)
	assert.NoError(suite.T(), err)
	assert.True(suite.T(), hasAccess)
}

func (suite *UserCanAccessTest) TestUserCanAccessRBACForbidden() {
	suite.mockRBAC.On("Validate", mock.Anything, mock.Anything).Return(models.ErrAccessForbidden)
	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(false, nil)

	hasAccess, err := suite.backend.UserCanAccess(suite.ctx, userID, extensionID)
	assert.Error(suite.T(), err)
	assert.Equal(suite.T(), models.ErrAccessForbidden, err)
	assert.False(suite.T(), hasAccess)
}

func (suite *UserCanAccessTest) TestUserCanAccessRBACError_PersonalExtension() {
	suite.mockRBAC.On("Validate", mock.Anything, mock.Anything).Return(errx.New(models.ErrInternalError))
	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(false, nil)
	suite.mockOwl.On("IsClientIDUserMatching", mock.Anything, userID, extensionID).Return(true, nil)

	hasAccess, err := suite.backend.UserCanAccess(suite.ctx, userID, extensionID)
	assert.NoError(suite.T(), err)
	assert.True(suite.T(), hasAccess)
}

func (suite *UserCanAccessTest) TestUserCanAccessRBACError_OrgExtension() {
	suite.mockRBAC.On("Validate", mock.Anything, mock.Anything).Return(errx.New(models.ErrInternalError))
	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(false, nil)
	suite.mockOwl.On("IsClientIDUserMatching", mock.Anything, userID, extensionID).Return(false, owl.ErrClientOwnedByOrg)

	hasAccess, err := suite.backend.UserCanAccess(suite.ctx, userID, extensionID)
	assert.Error(suite.T(), err)
	assert.Equal(suite.T(), models.ErrInternalError, errx.Unwrap(err))
	assert.False(suite.T(), hasAccess)
}

func TestUserCanAccessSuite(t *testing.T) {
	suite.Run(t, new(UserCanAccessTest))
}
