package games

import (
	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/insights/piper-service/backend"
	"code.justin.tv/insights/piper-service/backend/s3report"
	"code.justin.tv/insights/piper-service/internal/clients/discovery"
	"code.justin.tv/insights/piper-service/internal/clients/piperdb"
	"code.justin.tv/insights/piper-service/internal/clients/rbac"
	"code.justin.tv/insights/piper-service/internal/clients/s3"
	"code.justin.tv/insights/piper-service/internal/clients/uploader"
	"code.justin.tv/insights/piper-service/internal/clients/users"
	"code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

//go:generate mockery -name Backend
type Backend interface {
	GetGameReport(ctx context.Context, userID string, config config.PiperConfig) (*models.GameReports, error)
	GetReportTypes(ctx context.Context, userID string, gameID string, config config.PiperConfig) (*models.GameReportType, error)
	GetGameReportByDate(ctx context.Context, gameID, reportType, stateDate, endDate string, config config.PiperConfig) (string, error)
	GetAllReportDownloadURLs(ctx context.Context, userID, gameID, reportID, startDate, endDate string, limit, offset int, config config.PiperConfig) (models.HelixReports, *models.Pagination, error)
	TrackGamesReportUsage(ctx context.Context, gamesEvent models.SpadeEvent, config config.PiperConfig)
	UserCanAccess(ctx context.Context, domain string, userID string, gameID string, reportID string) (bool, error)
	CreateUploadServiceRequest(ctx context.Context, userID, gameID string, config config.PiperConfig) (*models.BoxArtUpload, error)
	UserOwnsGame(ctx context.Context, userID, gameID string) (bool, error)
	GetGameNameFromDiscovery(ctx context.Context, gameID string) (string, error)
	GetFittedDateRange(ctx context.Context, gameID, reportType, startDate, endDate string) (string, string, error)
}

type gamesRepoImpl struct {
	s3client        s3.Client
	cacher          backend.Cacher
	discoveryclient discovery.Client
	s3report        s3report.Backend
	spadeclient     spade.Client
	usersclient     users.Client
	piperdbclient   piperdb.Client
	uploaderclient  uploader.Client
	rbacClient      rbac.Client
}

func NewBackend(s3client s3.Client, cacher backend.Cacher, discoveryclient discovery.Client, spadeclient spade.Client, usersclient users.Client, s3report s3report.Backend, piperdbclient piperdb.Client, uploaderClient uploader.Client, rbacClient rbac.Client) (Backend, error) {
	return &gamesRepoImpl{
		s3client:        s3client,
		s3report:        s3report,
		cacher:          cacher,
		discoveryclient: discoveryclient,
		spadeclient:     spadeclient,
		usersclient:     usersclient,
		piperdbclient:   piperdbclient,
		uploaderclient:  uploaderClient,
		rbacClient:      rbacClient,
	}, nil
}

var _ Backend = new(gamesRepoImpl)
