package games

import (
	"fmt"
	"time"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/lib/reports"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

const (
	backendDomain           = "games"          // backendDomain is a Enum. its values can be "games", "extensions", "mods" and "one_pager". it's used in tracking tables and piper database tables.
	asyncTimeout            = 20 * time.Second // AsyncTimeout is the time we want to wait for async work to complete
	keyDelim                = "|"
	userID_games_key_prefix = "user_games_" // user "user_games_{userID}" to store <userID, gameIDs>
	urlExpireMinutes        = 5
	shortDate               = "2006-01-02"
)

// getGameNameFromDiscovery implemented with cache
func (c *gamesRepoImpl) GetGameNameFromDiscovery(ctx context.Context, gameID string) (string, error) {
	return c.discoveryclient.GetGameNameByID(ctx, gameID)
}

// UserCanAccess wrapps internal clients error and return ErrAccessForbidden to avoid exposing company, games and resource information to end users
func (c *gamesRepoImpl) UserCanAccess(ctx context.Context, domain string, userID string, gameID string, reportType string) (bool, error) {

	var err error

	// base line check: domain and report type.
	// report type can be empty; or if report type is given, it must be whitelisted in this domain
	if reportType != "" && !reports.IsWhitelisted(domain, reportType) {
		return false, models.ErrReportNotFound
	}

	if _, gameFound := models.BlackListedGames[gameID]; gameFound {
		logx.Warn(ctx, fmt.Sprintf("Game with ID %s is not allowed", gameID))
		return false, models.ErrAccessForbidden
	}

	// base line check: domain id
	_, err = c.GetGameNameFromDiscovery(ctx, gameID)
	if err != nil {
		logx.Error(ctx, fmt.Sprintf("Discovery failed to fetch game: %v", err))
		return false, models.ErrAccessForbidden
	}

	// staff account check: staff account can access any available reports for both visage and helix
	isStaff, err := c.usersclient.IsStaff(ctx, userID)
	if isStaff && err == nil {
		return true, nil
	} else if err != nil {
		logx.Warn(ctx, fmt.Sprintf("Couldn't retrieve user staff: %v", err))
	}

	// staff account check: Non-staff RBAC validate user resource will assures if an user can view a game
	// staff account check: If the report type is not specified, the request is from helix, non-staff user can access any whitelisted reports
	// staff account check: If the report type is specified, the request is from visage, non-staff user can only access the DevsiteGamesReportType report
	// non staff: check if company game is in the restricted list for one pager
	// non staff: RBAC check if user has permission to view a resource
	err = c.rbacClient.Validate(ctx, &rbacrpc.ValidateQuery{
		Permission:   "games::analytics::view",
		ResourceId:   gameID,
		ResourceType: "game",
		UserId:       userID,
	})

	if err != nil {
		logx.Warn(ctx, fmt.Sprintf("RBAC validation failed: %v", err))
		return false, models.ErrAccessForbidden
	}

	// non staff user can only access overview v2 in visage
	if reportType == "" || reportType == reports.DevsiteGamesReportType {
		return true, nil
	}

	// non staff has no access to this game
	return false, models.ErrAccessForbidden
}

// isValidGameID checks if gameID is valid
// empty string is always allowed
// otherwise check if the given gameID exists
func (c *gamesRepoImpl) isValidGameID(ctx context.Context, gameID string, cfg config.PiperConfig) error {
	if gameID != "" {
		_, err := c.GetGameNameFromDiscovery(ctx, gameID)
		if err != nil {
			return models.ErrGameNotFound
		}
	}
	return nil
}

// getValidUserGames assume gameID is valid
// if an user is staff, the user has access to all games.
// if the gameID is empty, return all games associate with this staff; otherwise return this gameID;
// if an user is non staff,  the user can only access his/her own games;
// if the gameID is not empty, check if the user has access; return gameID or user has no access to this resource error
func (c *gamesRepoImpl) getValidUserGames(ctx context.Context, userID, gameID string) ([]string, error) {
	// if there is an error when we check if an user is staff
	// we will log the error to rollbar and default this user as a non staff user
	isStaff, err := c.usersclient.IsStaff(ctx, userID)
	if err != nil {
		logx.Error(ctx, errx.Wrap(err, fmt.Sprintf("Couldn't retrieve user staff bit: %v", err)))
	}

	// staff can always access a single valid game
	if isStaff && gameID != "" {
		return []string{gameID}, nil
	}

	// for staff user and empty gameID, we return all the games this user can access;
	// for non staff user, we need to get all games this user has access to,
	// if gameID is not empty, we can if user can access, else return all games;
	gameIDs, err := c.rbacClient.GetGameIDsByUserID(ctx, userID)
	if gameIDs == nil || err != nil {
		return nil, errx.New(err)
	}

	var gameIDToReturn []string
	if isStaff {
		return gameIDs, nil
	} else if !isStaff && gameID == "" {
		for _, g := range gameIDs {
			gameIDToReturn = append(gameIDToReturn, g)
		}
		return gameIDToReturn, nil
	}

	for _, g := range gameIDs {
		if g == gameID {
			return []string{gameID}, nil
		}
	}

	return nil, models.ErrAccessForbidden
}
