package games

import (
	"fmt"
	"sort"
	"strings"
	"sync"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	cfg "code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/lib/reports"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

type goRoutineReport struct {
	report models.HelixReport
	err    error
}

func (c *gamesRepoImpl) GetAllReportDownloadURLs(ctx context.Context, userID, gameID, reportType, startDate, endDate string, limit, offset int, config cfg.PiperConfig) (models.HelixReports, *models.Pagination, error) {
	err := reports.IsValidLimit(limit)
	if err != nil {
		return nil, nil, err
	}

	// early check only one of the params is set, bad time format, and start is after end
	err = reports.IsValidDates(startDate, endDate)
	if err != nil {
		return nil, nil, err
	}

	// early check invalid report type
	err = reports.IsValidReportType(backendDomain, reportType)
	if err != nil {
		return nil, nil, err
	}

	// early check if the given game id is valid
	err = c.isValidGameID(ctx, gameID, config)
	if err != nil {
		return nil, nil, err
	}

	// check user role and game id and get games this user can access
	gameIDs, err := c.getValidUserGames(ctx, userID, gameID)
	if err != nil {
		return nil, nil, err
	}

	// set up game, report, start date, end date with valid value
	domainReportTypes := reports.GetValidReportTypes(backendDomain, reportType) // either reportType or all the types if reportType == ""
	var fullGameReports models.HelixReports
	for _, r := range domainReportTypes {
		for _, g := range gameIDs {
			fullGameReports = append(fullGameReports, models.HelixReport{
				ID:        g,
				Type:      r,
				StartDate: startDate,
				EndDate:   endDate,
			})
		}
	}

	if offset >= len(fullGameReports) {
		return nil, nil, models.ErrInvalidOffset
	}

	availableHelixReports, err := c.fetchAvailableHelixReports(ctx, userID, fullGameReports, config)
	// if there is no available Helix reports for the validated game id list, report list and date range lists combo,
	// if it's due to invalid date range or report not found error, return empty reports; otherwise return nil
	if len(availableHelixReports) == 0 {
		if errx.Cause(err) == models.ErrInvalidDateRange || errx.Cause(err) == models.ErrReportNotFound {
			return models.HelixReports{}, nil, errx.New(err)
		} else {
			return nil, nil, nil
		}
	}

	if offset >= len(availableHelixReports) {
		return nil, nil, models.ErrInvalidOffset
	}

	var gamesReports models.HelixReports
	for _, r := range availableHelixReports {
		gamesReports = append(gamesReports, r)
	}

	// always sort available Helix reports before we do pagination
	sort.Sort(gamesReports)

	pagination := &models.Pagination{Limit: limit, Offset: offset + limit}
	to := offset + limit
	if to >= len(gamesReports) {
		to = len(gamesReports)
		pagination = nil
	}

	gamesReports = gamesReports[offset:to]

	helixReports, err := c.fetchHelixReportsURL(ctx, userID, gamesReports, config)

	if len(helixReports) == 0 {
		return models.HelixReports{}, nil, errx.New(err)
	}

	return helixReports, pagination, nil
}

func recoverFetchReport(ctx context.Context, domainID string) {
	if r := recover(); r != nil {
		logx.Error(ctx, fmt.Sprintf("panic in async job to fetch Helix report for %s", domainID))
	}
}

func (c *gamesRepoImpl) fetchAvailableHelixReports(ctx context.Context, userID string, fullGameReports []models.HelixReport, config cfg.PiperConfig) (models.HelixReports, error) {
	var availableHelixReports []models.HelixReport

	var wg sync.WaitGroup
	resCh := make(chan *goRoutineReport, len(fullGameReports))

	for _, gameReport := range fullGameReports {
		bCtx, cancel := context.WithTimeout(context.Background(), asyncTimeout)

		wg.Add(1)

		go func(ctx context.Context, report models.HelixReport) {
			// defer recover to avoid a panic in a go routine to kill the service
			defer cancel()
			defer recoverFetchReport(ctx, report.ID)

			result, err := c.getReportDefaultDates(ctx, userID, report, config)
			resCh <- &goRoutineReport{
				result,
				err}

			wg.Done()
			return
		}(bCtx, gameReport)
	}

	wg.Wait()
	close(resCh)

	var fErr error
	for processedReport := range resCh {
		pErr := processedReport.err
		if pErr == nil && processedReport.report.StartDate != "" && processedReport.report.EndDate != "" {
			availableHelixReports = append(availableHelixReports, processedReport.report)
		} else if pErr != nil && errx.Cause(pErr) != models.ErrGameNotFound && errx.Cause(pErr) != models.ErrInvalidDateRange && errx.Cause(pErr) != models.ErrReportNotFound {
			// if a user queries multiple reports with the same date range, there may be reports hit dates out of range error
			// there may be game id does not have a game name
			// there may be no reports for a given game id and report type
			// we do not log these errors
			logx.Error(ctx, fmt.Sprintf("error getReportDefaultDates for user: %s report: %s. %v", userID, processedReport.report, pErr))
		} else if errx.Cause(pErr) != models.ErrGameNotFound {
			// if an error is due to out of bound dates, or no report exists for a game id and report type combo, we return any of this error for future check
			fErr = pErr
		}
	}

	return availableHelixReports, fErr
}

func (c *gamesRepoImpl) getReportDefaultDates(ctx context.Context, userID string, report models.HelixReport, config cfg.PiperConfig) (models.HelixReport, error) {
	key := userID + keyDelim + report.ID + keyDelim + report.Type + keyDelim + report.StartDate + keyDelim + report.EndDate

	prop, found := c.cacher.GetStringProperties(ctx, key)
	if found {
		defaultDates := strings.Split(prop, keyDelim)
		if len(defaultDates) == 2 {
			report.StartDate = defaultDates[0]
			report.EndDate = defaultDates[1]
			return report, nil
		}
	}

	reportStartDate, reportEndDate, err := c.GetFittedDateRange(ctx, report.ID, report.Type, report.StartDate, report.EndDate)
	if err != nil || reportStartDate == "" && reportEndDate == "" {
		return report, err
	}

	report.StartDate = reportStartDate
	report.EndDate = reportEndDate

	prop = reportStartDate + keyDelim + reportEndDate

	err = c.cacher.CacheStringProperties(ctx, key, prop)
	if err != nil {
		logx.Error(ctx, fmt.Sprintf("failed to find helix report dates for %s in cache", key))
	}

	return report, nil
}

func (c *gamesRepoImpl) fetchHelixReportsURL(ctx context.Context, userID string, gamesReports []models.HelixReport, config cfg.PiperConfig) (models.HelixReports, error) {
	var helixReports []models.HelixReport

	var wg sync.WaitGroup
	resCh := make(chan *goRoutineReport, len(gamesReports))

	for _, gameReport := range gamesReports {
		bCtx, cancel := context.WithTimeout(context.Background(), asyncTimeout)

		wg.Add(1)

		go func(ctx context.Context, report models.HelixReport) {
			// defer recover to avoid a panic in a go routine to kill the service
			defer cancel()
			defer recoverFetchReport(ctx, report.ID)

			url, err := c.generateReportURL(ctx, userID, report.ID, report.Type, report.StartDate, report.EndDate, config)
			report.URL = url

			resCh <- &goRoutineReport{
				report,
				err,
			}
			wg.Done()
		}(bCtx, gameReport)
	}

	wg.Wait()
	close(resCh)

	var fErr error
	for processedReport := range resCh {
		pErr := processedReport.err
		if pErr == nil && processedReport.report.URL != "" {
			helixReports = append(helixReports, processedReport.report)
		} else if pErr != nil && errx.Cause(pErr) != models.ErrGameNotFound {
			fErr = pErr
			logx.Error(ctx, fmt.Sprintf("error fetching URL for user: %s report: %s. %v", userID, processedReport.report, pErr))
		}
	}

	return helixReports, fErr
}
