package games

import (
	"testing"
	"time"

	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/insights/piper-service/backend/mocks"
	s3reportmock "code.justin.tv/insights/piper-service/backend/s3report/mocks"
	discoverymock "code.justin.tv/insights/piper-service/internal/clients/discovery/mocks"
	piperdbmock "code.justin.tv/insights/piper-service/internal/clients/piperdb/mocks"
	rbacmock "code.justin.tv/insights/piper-service/internal/clients/rbac/mocks"
	s3mock "code.justin.tv/insights/piper-service/internal/clients/s3/mocks"
	spademock "code.justin.tv/insights/piper-service/internal/clients/spade/mocks"
	uploadermock "code.justin.tv/insights/piper-service/internal/clients/uploader/mocks"
	usermock "code.justin.tv/insights/piper-service/internal/clients/users/mocks"
	piperconfig "code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

var (
	userID     = "123"
	userIDKey  = userID_games_key_prefix + userID
	overviewV1 = "overview_v1"
	overviewV2 = "overview_v2"
	dummyCfg   = piperconfig.PiperConfig{ReportBucket: "a_bucket", DynamicReportPrefix: "dynamic/"}
)

type GetAllURLsTest struct {
	suite.Suite
	mockS3        *s3mock.Client
	mockCacher    *mocks.Cacher
	mockDiscovery *discoverymock.Client
	mockSpade     *spademock.Client
	mockUsers     *usermock.Client
	mockS3Report  *s3reportmock.Backend
	mockPiperDB   *piperdbmock.Client
	mockUploader  *uploadermock.Client
	mockRBAC      *rbacmock.Client

	backend Backend
	config  *piperconfig.PiperConfig
	ctx     context.Context
}

func (suite *GetAllURLsTest) SetupTest() {
	var err error
	suite.mockS3 = &s3mock.Client{}
	suite.mockCacher = &mocks.Cacher{}
	suite.mockRBAC = &rbacmock.Client{}
	suite.mockSpade = &spademock.Client{}
	suite.mockUsers = &usermock.Client{}
	suite.mockDiscovery = &discoverymock.Client{}
	suite.mockS3Report = &s3reportmock.Backend{}
	suite.mockPiperDB = &piperdbmock.Client{}
	suite.mockUploader = &uploadermock.Client{}

	suite.backend, err = NewBackend(suite.mockS3, suite.mockCacher, suite.mockDiscovery, suite.mockSpade, suite.mockUsers, suite.mockS3Report, suite.mockPiperDB, suite.mockUploader, suite.mockRBAC)
	suite.config = &piperconfig.PiperConfig{ReportBucket: "a_bucket", DynamicReportPrefix: "dynamic/"}
	suite.ctx = context.Background()
	assert.NoError(suite.T(), err)
}

func (suite *GetAllURLsTest) TearDownTest() {
	suite.mockS3.AssertExpectations(suite.T())
	suite.mockCacher.AssertExpectations(suite.T())
	suite.mockSpade.AssertExpectations(suite.T())
	suite.mockUsers.AssertExpectations(suite.T())
	suite.mockDiscovery.AssertExpectations(suite.T())
	suite.mockS3Report.AssertExpectations(suite.T())
	suite.mockUploader.AssertExpectations(suite.T())
	suite.mockRBAC.AssertExpectations(suite.T())
}

func (suite *GetAllURLsTest) TestGetAllURLs_NoGameIDNoReportNoDates() {
	gameIDs := []string{"1212", "1213"}
	startDate := time.Now().UTC().AddDate(0, 0, -80).Format(shortDate)
	endDate := time.Now().UTC().AddDate(0, 0, -2).Format(shortDate)

	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(true, nil)

	suite.mockRBAC.On("GetGameIDsByUserID", mock.Anything, userID).Return(gameIDs, nil)
	suite.mockDiscovery.On("GetGameNameByID", mock.Anything, "1212").Return("Game 1", nil)
	suite.mockDiscovery.On("GetGameNameByID", mock.Anything, "1213").Return("Game 2", nil)

	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1212|"+overviewV1+"||").Return(startDate+"|"+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1212|"+overviewV2+"||").Return(startDate+"|"+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1213|"+overviewV1+"||").Return(startDate+"|"+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1213|"+overviewV2+"||").Return(startDate+"|"+endDate, true)

	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 1", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 1", overviewV2, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 2", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 2", overviewV2, startDate, endDate).Return([][]string{}, nil)

	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game 1", overviewV1, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game 1_overview_v1_2018-03-01_2018-04-01.csv", nil)
	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game 1", overviewV2, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game 1_overview_v2_2018-03-01_2018-04-01.csv", nil)
	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game 2", overviewV1, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game 2_overview_v1_2018-03-01_2018-04-01.csv", nil)
	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game 2", overviewV2, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game 2_overview_v2_2018-03-01_2018-04-01.csv", nil)

	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game 1_overview_v1_2018-03-01_2018-04-01.csv", "Game 1_overview_v1_2018-03-01.csv", urlExpireMinutes).Return("URL 1", nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game 1_overview_v2_2018-03-01_2018-04-01.csv", "Game 1_overview_v2_2018-03-01.csv", urlExpireMinutes).Return("URL 3", nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game 2_overview_v1_2018-03-01_2018-04-01.csv", "Game 2_overview_v1_2018-03-01.csv", urlExpireMinutes).Return("URL 2", nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game 2_overview_v2_2018-03-01_2018-04-01.csv", "Game 2_overview_v2_2018-03-01.csv", urlExpireMinutes).Return("URL 4", nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", "", "", "", 10, 0, dummyCfg)
	assert.NoError(suite.T(), err)
	assert.Nil(suite.T(), pagination)
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 4)
	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1212",
		Type:      overviewV1,
		URL:       "URL 1",
		StartDate: startDate,
		EndDate:   endDate,
	})
	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1212",
		Type:      overviewV2,
		URL:       "URL 3",
		StartDate: startDate,
		EndDate:   endDate,
	})
	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1213",
		Type:      overviewV1,
		URL:       "URL 2",
		StartDate: startDate,
		EndDate:   endDate,
	})
	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1213",
		Type:      overviewV2,
		URL:       "URL 4",
		StartDate: startDate,
		EndDate:   endDate,
	})
}

func (suite *GetAllURLsTest) TestGetAllURLs_GameIDNoReportNoDates() {

	startDate := time.Now().UTC().AddDate(0, 0, -80).Format(shortDate)
	endDate := time.Now().UTC().AddDate(0, 0, -2).Format(shortDate)

	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(true, nil)

	suite.mockDiscovery.On("GetGameNameByID", mock.Anything, "1212").Return("Game 1", nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game 1_overview_v1_2018-03-01_2018-04-01.csv", "Game 1_overview_v1_2018-03-01.csv", urlExpireMinutes).Return("URL 1", nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game 1_overview_v2_2018-03-01_2018-04-01.csv", "Game 1_overview_v2_2018-03-01.csv", urlExpireMinutes).Return("URL 2", nil)

	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game 1", overviewV1, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game 1_overview_v1_2018-03-01_2018-04-01.csv", nil)
	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game 1", overviewV2, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game 1_overview_v2_2018-03-01_2018-04-01.csv", nil)

	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 1", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 1", overviewV2, startDate, endDate).Return([][]string{}, nil)

	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1212|"+overviewV2+"||").Return(startDate+"|"+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1212|"+overviewV1+"||").Return(startDate+"|"+endDate, true)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "1212", "", "", "", 10, 0, dummyCfg)
	assert.NoError(suite.T(), err)
	assert.Nil(suite.T(), pagination)
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 2)
	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1212",
		Type:      overviewV1,
		URL:       "URL 1",
		StartDate: startDate,
		EndDate:   endDate,
	})
	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1212",
		Type:      overviewV2,
		URL:       "URL 2",
		StartDate: startDate,
		EndDate:   endDate,
	})
}

func (suite *GetAllURLsTest) TestGetAllURLs_GameIDReportNoDates() {

	startDate := time.Now().UTC().AddDate(0, 0, -80).Format(shortDate)
	endDate := time.Now().UTC().AddDate(0, 0, -2).Format(shortDate)

	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(true, nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game 1_overview_v1_2018-03-01_2018-04-01.csv", "Game 1_overview_v1_2018-03-01.csv", urlExpireMinutes).Return("URL 1", nil)
	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game 1", overviewV1, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game 1_overview_v1_2018-03-01_2018-04-01.csv", nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 1", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1212|"+overviewV1+"||").Return(startDate+"|"+endDate, true)

	suite.mockDiscovery.On("GetGameNameByID", mock.Anything, "1212").Return("Game 1", nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "1212", overviewV1, "", "", 10, 0, dummyCfg)
	assert.NoError(suite.T(), err)
	assert.Nil(suite.T(), pagination)
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 1)
	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1212",
		Type:      overviewV1,
		URL:       "URL 1",
		StartDate: startDate,
		EndDate:   endDate,
	})
}

func (suite *GetAllURLsTest) TestGetAllURLs_PageOne() {
	gameIDs := []string{"1212", "1213"}

	startDate := time.Now().UTC().AddDate(0, 0, -80).Format(shortDate)
	endDate := time.Now().UTC().AddDate(0, 0, -2).Format(shortDate)

	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(true, nil)

	suite.mockRBAC.On("GetGameIDsByUserID", mock.Anything, userID).Return(gameIDs, nil)

	suite.mockDiscovery.On("GetGameNameByID", mock.Anything, "1212").Return("Game 1", nil)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1212|"+overviewV1+"||").Return(startDate+"|"+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1212|"+overviewV2+"||").Return(startDate+"|"+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1213|"+overviewV1+"||").Return(startDate+"|"+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1213|"+overviewV2+"||").Return(startDate+"|"+endDate, true)

	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 1", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 1", overviewV2, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 2", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 2", overviewV2, startDate, endDate).Return([][]string{}, nil)

	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game 1", overviewV1, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game 1_overview_v1_2018-03-01_2018-04-01.csv", nil)

	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game 1_overview_v1_2018-03-01_2018-04-01.csv", "Game 1_overview_v1_2018-03-01.csv", urlExpireMinutes).Return("URL 1", nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", "", "", "", 1, 0, dummyCfg)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), pagination, &models.Pagination{Offset: 1, Limit: 1})
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 1)

	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1212",
		Type:      overviewV1,
		URL:       "URL 1",
		StartDate: startDate,
		EndDate:   endDate,
	})
}

func (suite *GetAllURLsTest) TestGetAllURLs_PageTwo() {
	gameIDs := []string{"1212", "1213"}

	startDate := time.Now().UTC().AddDate(0, 0, -80).Format(shortDate)
	endDate := time.Now().UTC().AddDate(0, 0, -2).Format(shortDate)

	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(true, nil)
	suite.mockRBAC.On("GetGameIDsByUserID", mock.Anything, userID).Return(gameIDs, nil)

	suite.mockDiscovery.On("GetGameNameByID", mock.Anything, "1212").Return("Game 1", nil)
	suite.mockDiscovery.On("GetGameNameByID", mock.Anything, "1213").Return("Game 2", nil)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1212|"+overviewV1+"||").Return(startDate+"|"+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1212|"+overviewV2+"||").Return(startDate+"|"+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1213|"+overviewV1+"||").Return(startDate+"|"+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1213|"+overviewV2+"||").Return(startDate+"|"+endDate, true)

	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 1", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 1", overviewV2, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 2", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 2", overviewV2, startDate, endDate).Return([][]string{}, nil)

	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game 1", overviewV2, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game 1_overview_v2_2018-03-01_2018-04-01.csv", nil)
	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game 2", overviewV1, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game 2_overview_v1_2018-03-01_2018-04-01.csv", nil)
	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game 2", overviewV2, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game 2_overview_v2_2018-03-01_2018-04-01.csv", nil)

	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game 1_overview_v2_2018-03-01_2018-04-01.csv", "Game 1_overview_v2_2018-03-01.csv", urlExpireMinutes).Return("URL 3", nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game 2_overview_v1_2018-03-01_2018-04-01.csv", "Game 2_overview_v1_2018-03-01.csv", urlExpireMinutes).Return("URL 2", nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game 2_overview_v2_2018-03-01_2018-04-01.csv", "Game 2_overview_v2_2018-03-01.csv", urlExpireMinutes).Return("URL 4", nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", "", "", "", 4, 1, dummyCfg)
	assert.NoError(suite.T(), err)
	assert.Nil(suite.T(), pagination)
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 3)

	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1213",
		Type:      overviewV1,
		URL:       "URL 2",
		StartDate: startDate,
		EndDate:   endDate,
	})
}

func (suite *GetAllURLsTest) TestGetAllURLs_BadOffset() {
	gameIDs := []string{"1212", "1213"}
	startDate := time.Now().UTC().AddDate(0, 0, -80).Format(shortDate)
	endDate := time.Now().UTC().AddDate(0, 0, -2).Format(shortDate)

	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(true, nil)

	suite.mockRBAC.On("GetGameIDsByUserID", mock.Anything, userID).Return(gameIDs, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 1", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 1", overviewV2, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 2", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 2", overviewV2, startDate, endDate).Return([][]string{}, nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", "", "", "", 1, 5, dummyCfg)
	assert.Equal(suite.T(), err, models.ErrInvalidOffset)
	assert.Nil(suite.T(), pagination)
	require.Nil(suite.T(), reports)
}

func (suite *GetAllURLsTest) TestGetAllURLs_BadLimit() {
	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, "123", "", "", "", "", 0, 3, dummyCfg)
	assert.Equal(suite.T(), err, models.ErrInvalidLimit)
	assert.Nil(suite.T(), pagination)
	require.Nil(suite.T(), reports)
}

func (suite *GetAllURLsTest) TestGetAllURLs_BadTimeFormat() {

	badStartDate := "2017a23:00:00T"
	badEndDate := time.Now().UTC().AddDate(0, 0, 1).Format(time.RFC3339)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "1212", overviewV1, badStartDate, badEndDate, 1, 0, dummyCfg)
	assert.Equal(suite.T(), errx.New(models.ErrInvalidDateFormat).Error(), err.Error())
	assert.Nil(suite.T(), pagination)
	assert.Nil(suite.T(), reports)
}

func (suite *GetAllURLsTest) TestGetAllURLs_InvalidDateRange() {
	var startDate string
	var endDate string
	var err error

	// dates not allowed, it needs to be a RFC3339 timestamp
	startDate = "2018-07-22"
	endDate = "2018-07-24"
	_, _, err = suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", "", startDate, endDate, 10, 0, piperconfig.PiperConfig{})
	suite.EqualError(err, models.ErrInvalidDateFormat.Error())

	// start date needs to be before end date
	startDate = "2018-07-24T00:00:00Z"
	endDate = "2018-07-23T00:00:00Z"
	_, _, err = suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", "", startDate, endDate, 10, 0, piperconfig.PiperConfig{})
	suite.EqualError(err, models.ErrInvalidDateRange.Error())

	// need both start and end date
	startDate = "2018-07-24T00:00:00Z"
	endDate = ""
	_, _, err = suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", "", startDate, endDate, 10, 0, piperconfig.PiperConfig{})
	suite.EqualError(err, models.ErrInvalidDateFormat.Error())
}

func (suite *GetAllURLsTest) TestGetAllURLs_OneInvalidDateRange() {
	gameIDs := []string{"1212"}

	d := time.Date(2018, 7, 22, 0, 0, 0, 0, time.UTC)
	startDate := d.AddDate(0, 0, -300).Format(time.RFC3339)
	endDate := d.AddDate(0, 0, -100).Format(time.RFC3339)
	v1dbStartDate := d.AddDate(0, 0, -80)
	v1dbEndDate := d.AddDate(0, 0, -2)
	v2StartDate := d.AddDate(0, 0, -300).Format(shortDate)
	v2EndDate := d.AddDate(0, 0, -100).Format(shortDate)

	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(true, nil)
	suite.mockRBAC.On("GetGameIDsByUserID", mock.Anything, userID).Return(gameIDs, nil)

	suite.mockDiscovery.On("GetGameNameByID", mock.Anything, "1212").Return("Game 1", nil)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1212|"+overviewV1+"|"+startDate+"|"+endDate).Return("", false)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1212|"+overviewV2+"|"+startDate+"|"+endDate).Return(v2StartDate+"|"+v2EndDate, true)

	suite.mockPiperDB.On("GetDomainReportDateRange", mock.Anything, backendDomain, overviewV1, "1212").Return(v1dbStartDate, v1dbEndDate, nil)

	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game 1", overviewV2, v2StartDate, v2EndDate).Return([][]string{}, nil)

	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game 1", overviewV2, v2StartDate, v2EndDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game 1_overview_v2_2018-03-01_2018-04-01.csv", nil)
	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game 1_overview_v2_2018-03-01_2018-04-01.csv", "Game 1_overview_v2_2018-03-01.csv", urlExpireMinutes).Return("URL 3", nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", "", startDate, endDate, 10, 0, dummyCfg)
	assert.NoError(suite.T(), err)
	assert.Nil(suite.T(), pagination)
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 1)

	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1212",
		Type:      overviewV2,
		URL:       "URL 3",
		StartDate: v2StartDate,
		EndDate:   v2EndDate,
	})
}

func (suite *GetAllURLsTest) TestGetAllURLs_InvalidReportType() {
	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "", "notype", "", "", 1, 0, dummyCfg)
	assert.Equal(suite.T(), errx.New(models.ErrInvalidReportType).Error(), err.Error())
	assert.Nil(suite.T(), reports)
	assert.Nil(suite.T(), pagination)
}

func (suite *GetAllURLsTest) TestGetAllURLs_ValidGameIDNonStaff() {
	gameIDs := []string{"1212"}

	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(false, nil)
	suite.mockRBAC.On("GetGameIDsByUserID", mock.Anything, userID).Return(gameIDs, nil)
	suite.mockDiscovery.On("GetGameNameByID", mock.Anything, "1111").Return("Game 1", nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "1111", "", "", "", 10, 0, dummyCfg)
	assert.Equal(suite.T(), errx.New(models.ErrAccessForbidden).Error(), err.Error())
	assert.Nil(suite.T(), reports)
	assert.Nil(suite.T(), pagination)
}

func (suite *GetAllURLsTest) TestGetAllURLs_ValidGameIDStaff() {

	startDate := time.Now().UTC().AddDate(0, 0, -80).Format(shortDate)
	endDate := time.Now().UTC().AddDate(0, 0, -2).Format(shortDate)

	suite.mockS3.On("PreSignURL", "a_bucket", "dynamic/Game1111_overview_v1_2018-03-01_2018-04-01.csv", "Game1111_overview_v1_2018-03-01.csv", urlExpireMinutes).Return("URL1111", nil)

	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game1111", overviewV1, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game1111_overview_v1_2018-03-01_2018-04-01.csv", nil)
	suite.mockS3Report.On("SaveReportToS3", mock.Anything, [][]string{}, "Game1111", overviewV2, startDate, endDate, dummyCfg.ReportBucket, dummyCfg.DynamicReportPrefix).Return("Game1111_overview_v1_2018-03-01_2018-04-01.csv", nil)

	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game1111", overviewV1, startDate, endDate).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetGameReportByDate", mock.Anything, "Game1111", overviewV2, startDate, endDate).Return([][]string{}, nil)

	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1111|"+overviewV1+"||").Return(startDate+"|"+endDate, true)
	suite.mockCacher.On("GetStringProperties", mock.Anything, userID+"|1111|"+overviewV2+"||").Return(startDate+"|"+endDate, true)

	suite.mockDiscovery.On("GetGameNameByID", mock.Anything, "1111").Return("Game1111", nil)

	suite.mockUsers.On("IsStaff", mock.Anything, userID).Return(true, nil)

	reports, pagination, err := suite.backend.GetAllReportDownloadURLs(suite.ctx, userID, "1111", "", "", "", 10, 0, dummyCfg)
	assert.NoError(suite.T(), err)
	assert.Nil(suite.T(), pagination)
	require.NotNil(suite.T(), reports)
	require.Len(suite.T(), reports, 2)

	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1111",
		Type:      overviewV1,
		URL:       "URL1111",
		StartDate: startDate,
		EndDate:   endDate,
	})
	assert.Contains(suite.T(), reports, models.HelixReport{
		ID:        "1111",
		Type:      overviewV2,
		URL:       "URL1111",
		StartDate: startDate,
		EndDate:   endDate,
	})
}

func TestGetAllURLsSuite(t *testing.T) {
	suite.Run(t, new(GetAllURLsTest))
}
