package games

import (
	"testing"

	"code.justin.tv/insights/piper-service/lib/reports"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"

	"code.justin.tv/insights/piper-service/backend/mocks"
	s3reportmock "code.justin.tv/insights/piper-service/backend/s3report/mocks"
	discoverymock "code.justin.tv/insights/piper-service/internal/clients/discovery/mocks"
	piperdbmock "code.justin.tv/insights/piper-service/internal/clients/piperdb/mocks"
	rbacmock "code.justin.tv/insights/piper-service/internal/clients/rbac/mocks"
	s3mock "code.justin.tv/insights/piper-service/internal/clients/s3/mocks"
	spademock "code.justin.tv/insights/piper-service/internal/clients/spade/mocks"
	uploadermock "code.justin.tv/insights/piper-service/internal/clients/uploader/mocks"
	usermock "code.justin.tv/insights/piper-service/internal/clients/users/mocks"
	piperconfig "code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/models"
)

type GetReportTypesTest struct {
	suite.Suite
	mockS3        *s3mock.Client
	mockCacher    *mocks.Cacher
	mockDiscovery *discoverymock.Client
	mockSpade     *spademock.Client
	mockUsers     *usermock.Client
	mockS3Report  *s3reportmock.Backend
	mockPiperDB   *piperdbmock.Client
	mockUploader  *uploadermock.Client
	mockRBAC      *rbacmock.Client

	backend Backend
	config  *piperconfig.PiperConfig
	ctx     context.Context
}

func (suite *GetReportTypesTest) SetupTest() {
	var err error
	suite.mockS3 = &s3mock.Client{}
	suite.mockCacher = &mocks.Cacher{}
	suite.mockSpade = &spademock.Client{}
	suite.mockUsers = &usermock.Client{}
	suite.mockDiscovery = &discoverymock.Client{}
	suite.mockS3Report = &s3reportmock.Backend{}
	suite.mockUploader = &uploadermock.Client{}
	suite.mockRBAC = &rbacmock.Client{}
	suite.mockPiperDB = &piperdbmock.Client{}

	suite.backend, err = NewBackend(suite.mockS3, suite.mockCacher, suite.mockDiscovery, suite.mockSpade, suite.mockUsers, suite.mockS3Report, suite.mockPiperDB, suite.mockUploader, suite.mockRBAC)
	suite.config = &piperconfig.PiperConfig{ReportBucket: "", ReportPrefix: ""}
	suite.ctx = context.Background()
	assert.NoError(suite.T(), err)
}

func (suite *GetReportTypesTest) TearDownTest() {
	suite.mockS3.AssertExpectations(suite.T())
	suite.mockCacher.AssertExpectations(suite.T())
	suite.mockSpade.AssertExpectations(suite.T())
	suite.mockUsers.AssertExpectations(suite.T())
	suite.mockDiscovery.AssertExpectations(suite.T())
	suite.mockS3Report.AssertExpectations(suite.T())
	suite.mockUploader.AssertExpectations(suite.T())
	suite.mockRBAC.AssertExpectations(suite.T())
}

func (suite *GetReportTypesTest) TestGetReportTypesGameHasReport() {
	gameReportType := models.GameReportType{GameID: nonStaffUserOwnedGameID, ReportTypes: []string{reports.DevsiteGamesReportType}}
	gameIDs := []string{nonStaffUserOwnedGameID}

	suite.mockPiperDB.On("GetAvailableDomainsWithReports", suite.ctx, backendDomain, reports.DevsiteGamesReportType, gameIDs).Return(gameIDs, nil)
	result, err := suite.backend.GetReportTypes(suite.ctx, nonStaffUserID, nonStaffUserOwnedGameID, *suite.config)
	assert.Nil(suite.T(), err)
	require.NotNil(suite.T(), result)
	assert.Equal(suite.T(), &gameReportType, result)
}

func (suite *GetReportTypesTest) TestGetReportTypesGameHasNoReport() {
	gameReportType := models.GameReportType{GameID: nonStaffUserOwnedGameID, ReportTypes: []string{}}
	gameIDs := []string{nonStaffUserOwnedGameID}

	suite.mockPiperDB.On("GetAvailableDomainsWithReports", suite.ctx, backendDomain, reports.DevsiteGamesReportType, gameIDs).Return([]string{}, nil)
	result, err := suite.backend.GetReportTypes(suite.ctx, nonStaffUserID, nonStaffUserOwnedGameID, *suite.config)
	assert.Nil(suite.T(), err)
	require.NotNil(suite.T(), result)
	assert.Equal(suite.T(), &gameReportType, result)
}

func (suite *GetReportTypesTest) TestGetReportTypesDBError() {
	gameIDs := []string{nonStaffUserOwnedGameID}

	suite.mockPiperDB.On("GetAvailableDomainsWithReports", suite.ctx, backendDomain, reports.DevsiteGamesReportType, gameIDs).Return([]string{}, models.ErrReportNotFound)
	result, err := suite.backend.GetReportTypes(suite.ctx, nonStaffUserID, nonStaffUserOwnedGameID, *suite.config)
	require.Equal(suite.T(), models.ErrReportNotFound, err)
	assert.Nil(suite.T(), result)
}

func TestRunGetReportTypesSuite(t *testing.T) {
	suite.Run(t, new(GetReportTypesTest))
}
