package games

import (
	"testing"

	"code.justin.tv/insights/piper-service/backend/mocks"
	s3reportmock "code.justin.tv/insights/piper-service/backend/s3report/mocks"
	discoverymock "code.justin.tv/insights/piper-service/internal/clients/discovery/mocks"
	piperdbmock "code.justin.tv/insights/piper-service/internal/clients/piperdb/mocks"
	rbacmock "code.justin.tv/insights/piper-service/internal/clients/rbac/mocks"
	s3mock "code.justin.tv/insights/piper-service/internal/clients/s3/mocks"
	spademock "code.justin.tv/insights/piper-service/internal/clients/spade/mocks"
	uploadermock "code.justin.tv/insights/piper-service/internal/clients/uploader/mocks"
	usermock "code.justin.tv/insights/piper-service/internal/clients/users/mocks"
	piperconfig "code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/lib/reports"
	"code.justin.tv/insights/piper-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

type GetReportsTest struct {
	suite.Suite
	mockS3        *s3mock.Client
	mockCacher    *mocks.Cacher
	mockDiscovery *discoverymock.Client
	mockSpade     *spademock.Client
	mockUsers     *usermock.Client
	mockS3Report  *s3reportmock.Backend
	mockPiperDB   *piperdbmock.Client
	mockUploader  *uploadermock.Client
	mockRBAC      *rbacmock.Client

	backend Backend
	config  *piperconfig.PiperConfig
	ctx     context.Context
}

func (suite *GetReportsTest) SetupTest() {
	var err error
	suite.mockS3 = &s3mock.Client{}
	suite.mockCacher = &mocks.Cacher{}
	suite.mockRBAC = &rbacmock.Client{}
	suite.mockSpade = &spademock.Client{}
	suite.mockUsers = &usermock.Client{}
	suite.mockDiscovery = &discoverymock.Client{}
	suite.mockS3Report = &s3reportmock.Backend{}
	suite.mockPiperDB = &piperdbmock.Client{}
	suite.mockUploader = &uploadermock.Client{}

	suite.backend, err = NewBackend(suite.mockS3, suite.mockCacher, suite.mockDiscovery, suite.mockSpade, suite.mockUsers, suite.mockS3Report, suite.mockPiperDB, suite.mockUploader, suite.mockRBAC)
	suite.config = &piperconfig.PiperConfig{ReportBucket: "a_bucket", DynamicReportPrefix: "dynamic/"}
	suite.ctx = context.Background()
	assert.NoError(suite.T(), err)
}

func (suite *GetReportsTest) TearDownTest() {
	suite.mockS3.AssertExpectations(suite.T())
	suite.mockCacher.AssertExpectations(suite.T())
	suite.mockSpade.AssertExpectations(suite.T())
	suite.mockUsers.AssertExpectations(suite.T())
	suite.mockDiscovery.AssertExpectations(suite.T())
	suite.mockS3Report.AssertExpectations(suite.T())
	suite.mockUploader.AssertExpectations(suite.T())
	suite.mockRBAC.AssertExpectations(suite.T())
}

func (suite *GetReportsTest) TestGetGameReportWithGames() {
	gameIDs := []string{"123", "321"}
	expected := models.GameReports{}
	expected = append(expected, models.GameReport{GameID: "123", ReportName: reports.DevsiteGamesReportType})

	suite.mockPiperDB.On("GetAvailableDomainsWithReports", suite.ctx, backendDomain, reports.DevsiteGamesReportType, gameIDs).Return([]string{"123"}, nil)
	suite.mockRBAC.On("GetGameIDsByUserID", suite.ctx, userID).Return(gameIDs, nil)
	result, err := suite.backend.GetGameReport(suite.ctx, userID, dummyCfg)
	assert.NoError(suite.T(), err)
	require.Equal(suite.T(), &expected, result)
}

func (suite *GetReportsTest) TestGetGameReportNoGameNoErr() {
	gameIDs := []string{"123", "321"}
	expected := models.GameReports{}

	suite.mockPiperDB.On("GetAvailableDomainsWithReports", suite.ctx, backendDomain, reports.DevsiteGamesReportType, gameIDs).Return([]string{}, nil)
	suite.mockRBAC.On("GetGameIDsByUserID", suite.ctx, userID).Return(gameIDs, nil)
	result, err := suite.backend.GetGameReport(suite.ctx, userID, dummyCfg)
	assert.NoError(suite.T(), err)
	require.Equal(suite.T(), &expected, result)
}

func TestGetReportsSuite(t *testing.T) {
	suite.Run(t, new(GetReportsTest))
}
