package games

import (
	"testing"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/insights/piper-service/lib/reports"
	"code.justin.tv/insights/piper-service/models"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"

	"code.justin.tv/insights/piper-service/backend/mocks"
	s3reportmock "code.justin.tv/insights/piper-service/backend/s3report/mocks"
	discoverymock "code.justin.tv/insights/piper-service/internal/clients/discovery/mocks"
	piperdbmock "code.justin.tv/insights/piper-service/internal/clients/piperdb/mocks"
	rbacmock "code.justin.tv/insights/piper-service/internal/clients/rbac/mocks"
	s3mock "code.justin.tv/insights/piper-service/internal/clients/s3/mocks"
	spademock "code.justin.tv/insights/piper-service/internal/clients/spade/mocks"
	uploadermock "code.justin.tv/insights/piper-service/internal/clients/uploader/mocks"
	usermock "code.justin.tv/insights/piper-service/internal/clients/users/mocks"
	piperconfig "code.justin.tv/insights/piper-service/internal/config"
)

var (
	staffUserID                  = "12345"
	staffUserOwnedGameID         = "111111"
	staffUserOwnedGameName       = "game111111"
	staffUserNotOwnedGameID      = "444444"
	staffUserNotOwnedGameName    = "game444444"
	nonStaffUserID               = "23456"
	nonStaffUserOwnedGameID      = "222222"
	nonStaffUserOwnedGameName    = "game222222"
	nonStaffUserNotOwnedGameID   = "333333"
	nonStaffUserNotOwnedGameName = "game333333"
	inValidGameID                = "001"
	whilteListReportType         = reports.DevsiteGamesReportType
	userCompany                  = rbacrpc.Company{Id: "good_company"}
)

type GetTest struct {
	suite.Suite
	mockS3        *s3mock.Client
	mockCacher    *mocks.Cacher
	mockDiscovery *discoverymock.Client
	mockSpade     *spademock.Client
	mockUsers     *usermock.Client
	mockS3Report  *s3reportmock.Backend
	mockPiperDB   *piperdbmock.Client
	mockUploader  *uploadermock.Client
	mockRBAC      *rbacmock.Client

	backend Backend
	config  *piperconfig.PiperConfig
	ctx     context.Context
}

func (suite *GetTest) SetupTest() {
	var err error
	suite.mockS3 = &s3mock.Client{}
	suite.mockCacher = &mocks.Cacher{}
	suite.mockSpade = &spademock.Client{}
	suite.mockUsers = &usermock.Client{}
	suite.mockDiscovery = &discoverymock.Client{}
	suite.mockS3Report = &s3reportmock.Backend{}
	suite.mockUploader = &uploadermock.Client{}
	suite.mockRBAC = &rbacmock.Client{}

	suite.backend, err = NewBackend(suite.mockS3, suite.mockCacher, suite.mockDiscovery, suite.mockSpade, suite.mockUsers, suite.mockS3Report, suite.mockPiperDB, suite.mockUploader, suite.mockRBAC)
	suite.config = &piperconfig.PiperConfig{ReportBucket: "", ReportPrefix: ""}
	suite.ctx = context.Background()
	assert.NoError(suite.T(), err)
}

func (suite *GetTest) TearDownTest() {
	suite.mockS3.AssertExpectations(suite.T())
	suite.mockCacher.AssertExpectations(suite.T())
	suite.mockSpade.AssertExpectations(suite.T())
	suite.mockUsers.AssertExpectations(suite.T())
	suite.mockDiscovery.AssertExpectations(suite.T())
	suite.mockS3Report.AssertExpectations(suite.T())
	suite.mockUploader.AssertExpectations(suite.T())
	suite.mockRBAC.AssertExpectations(suite.T())
}

func (suite *GetTest) TestUserCanAccess_NonStaffUserOwnsAGame() {
	suite.mockRBAC.On("Validate", suite.ctx, mock.Anything).Return(nil)
	suite.mockUsers.On("IsStaff", suite.ctx, nonStaffUserID).Return(false, nil)
	suite.mockDiscovery.On("GetGameNameByID", suite.ctx, nonStaffUserOwnedGameID).Return(nonStaffUserOwnedGameName, nil)

	result, err := suite.backend.UserCanAccess(suite.ctx, backendDomain, nonStaffUserID, nonStaffUserOwnedGameID, whilteListReportType)
	assert.Nil(suite.T(), err)
	require.NotNil(suite.T(), result)
	assert.Equal(suite.T(), true, result)
}

func (suite *GetTest) TestUserCanAccess_NonStaffUserNotOwnsTheGame() {

	suite.mockRBAC.On("Validate", suite.ctx, mock.Anything).Return(models.ErrAccessForbidden)
	suite.mockUsers.On("IsStaff", suite.ctx, nonStaffUserID).Return(false, nil)
	suite.mockDiscovery.On("GetGameNameByID", suite.ctx, nonStaffUserNotOwnedGameID).Return(nonStaffUserNotOwnedGameName, nil)

	result, err := suite.backend.UserCanAccess(suite.ctx, backendDomain, nonStaffUserID, nonStaffUserNotOwnedGameID, whilteListReportType)
	assert.Equal(suite.T(), false, result)
	assert.Equal(suite.T(), err, models.ErrAccessForbidden)
}

func (suite *GetTest) TestUserCanAccess_StaffUserNotOwnsTheGame() {

	suite.mockUsers.On("IsStaff", suite.ctx, staffUserID).Return(true, nil)
	suite.mockDiscovery.On("GetGameNameByID", suite.ctx, staffUserNotOwnedGameID).Return(staffUserNotOwnedGameName, nil)

	result, err := suite.backend.UserCanAccess(suite.ctx, backendDomain, staffUserID, staffUserNotOwnedGameID, whilteListReportType)
	assert.Nil(suite.T(), err)
	require.NotNil(suite.T(), result)
	assert.Equal(suite.T(), true, result)
}

func (suite *GetTest) TestUserCanAccess_InvalidGame() {

	suite.mockDiscovery.On("GetGameNameByID", suite.ctx, inValidGameID).Return("", models.ErrGameNotFound)

	result, err := suite.backend.UserCanAccess(suite.ctx, backendDomain, staffUserID, inValidGameID, whilteListReportType)
	assert.Equal(suite.T(), models.ErrAccessForbidden, err)
	assert.Equal(suite.T(), false, result)
}

func (suite *GetTest) TestUserCanAccess_BlacklistedGame_Staff() {
	result, err := suite.backend.UserCanAccess(suite.ctx, backendDomain, staffUserID, "509660", whilteListReportType)
	assert.Equal(suite.T(), models.ErrAccessForbidden, err)
	assert.Equal(suite.T(), false, result)
}

func (suite *GetTest) TestUserCanAccess_BlacklistedGame_NonStaff() {
	result, err := suite.backend.UserCanAccess(suite.ctx, backendDomain, nonStaffUserID, "509660", whilteListReportType)
	assert.Equal(suite.T(), models.ErrAccessForbidden, err)
	assert.Equal(suite.T(), false, result)
}

func (suite *GetTest) TestUserCanAccess_NonStaffUserDoesNotHaveViewPermission() {
	suite.mockDiscovery.On("GetGameNameByID", suite.ctx, nonStaffUserOwnedGameID).Return(mock.Anything, nil)

	suite.mockRBAC.On("Validate", suite.ctx, mock.Anything).Return(models.ErrAccessForbidden)
	suite.mockUsers.On("IsStaff", suite.ctx, nonStaffUserID).Return(false, nil)

	result, err := suite.backend.UserCanAccess(suite.ctx, backendDomain, nonStaffUserID, nonStaffUserOwnedGameID, whilteListReportType)
	require.Equal(suite.T(), models.ErrAccessForbidden, err)
	assert.Equal(suite.T(), false, result)
}

func TestRunGetSuite(t *testing.T) {
	suite.Run(t, new(GetTest))
}
