package games

import (
	"testing"

	cachermock "code.justin.tv/insights/piper-service/backend/mocks"
	s3reportmock "code.justin.tv/insights/piper-service/backend/s3report/mocks"
	discoverymock "code.justin.tv/insights/piper-service/internal/clients/discovery/mocks"
	emsmock "code.justin.tv/insights/piper-service/internal/clients/ems/mocks"
	piperdbmock "code.justin.tv/insights/piper-service/internal/clients/piperdb/mocks"
	rbacmock "code.justin.tv/insights/piper-service/internal/clients/rbac/mocks"
	s3mock "code.justin.tv/insights/piper-service/internal/clients/s3/mocks"
	spademock "code.justin.tv/insights/piper-service/internal/clients/spade/mocks"
	uploadermock "code.justin.tv/insights/piper-service/internal/clients/uploader/mocks"
	usersmock "code.justin.tv/insights/piper-service/internal/clients/users/mocks"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

type UserOwnsGameTest struct {
	suite.Suite

	piperDBClient   *piperdbmock.Client
	discoveryClient *discoverymock.Client
	spadeClient     *spademock.Client
	rbacClient      *rbacmock.Client
	usersClient     *usersmock.Client
	emsClient       *emsmock.Client
	s3Client        *s3mock.Client
	s3report        *s3reportmock.Backend
	mockCacher      *cachermock.Cacher
	mockUploader    *uploadermock.Client

	gameImpl *gamesRepoImpl

	userID      string
	gameID      string
	companyIDA  string
	companyIDB  string
	companyName string

	noGameUser string
	ctx        context.Context
}

func (suite *UserOwnsGameTest) SetupTest() {
	var err error

	suite.piperDBClient = &piperdbmock.Client{}
	suite.discoveryClient = &discoverymock.Client{}
	suite.spadeClient = &spademock.Client{}
	suite.rbacClient = &rbacmock.Client{}
	suite.usersClient = &usersmock.Client{}
	suite.emsClient = &emsmock.Client{}
	suite.s3Client = &s3mock.Client{}
	suite.s3report = &s3reportmock.Backend{}
	suite.mockCacher = &cachermock.Cacher{}
	suite.mockUploader = &uploadermock.Client{}

	suite.gameImpl = &gamesRepoImpl{
		suite.s3Client,
		suite.mockCacher,
		suite.discoveryClient,
		suite.s3report,
		suite.spadeClient,
		suite.usersClient,
		suite.piperDBClient,
		suite.mockUploader,
		suite.rbacClient,
	}
	suite.NoError(err)

	suite.userID = "131180911"
	suite.gameID = "488552"

	suite.noGameUser = "270724749"
	suite.ctx = context.Background()
}

func (suite *UserOwnsGameTest) TestDeveloperCanAccessOwnResource() {
	suite.usersClient.On("IsStaff", suite.ctx, suite.userID).Return(false, nil)
	suite.rbacClient.On("Validate", suite.ctx, mock.Anything).Return(nil)

	ifCanAccess, err := suite.gameImpl.UserOwnsGame(suite.ctx, suite.userID, suite.gameID)
	suite.NoError(err)
	suite.True(ifCanAccess)
}

func (suite *UserOwnsGameTest) TestUserCantAccessWhenError() {
	suite.usersClient.On("IsStaff", suite.ctx, suite.userID).Return(false, nil)
	suite.rbacClient.On("Validate", suite.ctx, mock.Anything).Return(errors.New("unexpected error"))

	ifCanAccess, err := suite.gameImpl.UserOwnsGame(suite.ctx, suite.userID, suite.gameID)
	suite.Error(err)
	suite.False(ifCanAccess)
}

func (suite *UserOwnsGameTest) TestStaffHasSuperAdminPermission() {
	suite.usersClient.On("IsStaff", suite.ctx, suite.userID).Return(true, nil)

	ifCanAccess, err := suite.gameImpl.UserOwnsGame(suite.ctx, suite.userID, suite.gameID)
	suite.NoError(err)
	suite.True(ifCanAccess)
}

func TestUserCanAccessSuite(t *testing.T) {
	suite.Run(t, new(UserOwnsGameTest))
}
