package mods

import (
	"fmt"
	"strconv"

	"code.justin.tv/chat/golibs/logx"
	piperconfig "code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

const (
	backendDomain       = "mods" // backendDomain is a Enum. its values can be "games", "extensions", "mods" and "one_pager". it's used in tracking tables and piper database tables.
	urlExpireMinutes    = 5
	getModNamePrefix    = "getModName"
	getModNameDelimeter = "|"
)

func (c *modsImpl) UserCanAccess(ctx context.Context, userID, modID string, config piperconfig.PiperConfig) (bool, *models.Mod, error) {
	mods, err := c.eleriumClient.GetModsByUserID(ctx, userID, config)
	if err != nil || mods == nil {
		return false, nil, err
	}

	// in order to return the mod associated to the given mod id
	// we find the matched mod first then check if the user is staff
	for _, m := range *mods {
		if strconv.Itoa(m.Id) == modID {
			return true, &m, nil
		}
	}

	isStaff, err := c.usersClient.IsStaff(ctx, userID)
	if err != nil {
		logx.Error(ctx, fmt.Sprintf("Couldn't retrieve user staff info: %v", err))
	}
	if isStaff {
		return true, nil, nil
	}

	return false, nil, nil
}

func (c *modsImpl) getModNameFromDB(ctx context.Context, modID string, config piperconfig.PiperConfig) (string, error) {
	key := getModNamePrefix + getModNameDelimeter + modID

	modName, found := c.cacher.GetStringProperties(ctx, key)
	if found {
		return modName, nil
	}

	modName, err := c.piperdbclient.GetModName(ctx, modID)
	if err != nil {
		return "", models.ErrResourceNotFound
	}

	err = c.cacher.CacheStringProperties(ctx, key, modName)
	if err != nil {
		logx.Error(ctx, fmt.Sprintf("failed to call GetModName for %s in cache", key))
	}

	return modName, nil
}
