package mods

import (
	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/insights/piper-service/models"

	piperconfig "code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/lib/reports"
	"golang.org/x/net/context"
)

func (c *modsImpl) GetModReportURL(ctx context.Context, userID, modID, reportType string, conf piperconfig.PiperConfig) (string, error) {
	if !IsValidReportType(reportType) {
		return "", models.ErrInvalidReportType
	}

	userCanAccess, mod, err := c.UserCanAccess(ctx, userID, modID, conf)
	if err != nil {
		return "", err
	}

	if !userCanAccess {
		return "", models.ErrAccessForbidden
	}
	var modName string
	if mod == nil || mod.Name == "" {
		modName, err = c.getModNameFromDB(ctx, modID, conf)
		if err != nil {
			return "", err
		}
	} else {
		modName = mod.Name
	}
	return c.getReportByDate(ctx, userID, modID, modName, reportType, "", "", conf)
}

func (c *modsImpl) getReportByDate(ctx context.Context, userID, modID, modName, reportyType, startDate, endDate string, conf piperconfig.PiperConfig) (string, error) {
	startDate, endDate, err := c.GetFittedDateRange(ctx, modID, reportyType, startDate, endDate)
	if err != nil {
		return "", errx.New(err)
	}

	return c.generateReportURL(ctx, userID, modID, reportyType, startDate, endDate, modName, conf)
}

func (c *modsImpl) generateReportURL(ctx context.Context, userID, modID, reportType, startDate, endDate, modName string, conf piperconfig.PiperConfig) (string, error) {
	report, err := c.piperdbclient.GetModReportByDate(ctx, modID, reportType, startDate, endDate)

	if err != nil {
		return "", errx.New(err)
	}

	fileName, err := c.s3Report.SaveReportToS3(ctx, report, modName, reportType, startDate, endDate, conf.ReportBucket, conf.DynamicReportPrefix)
	if err != nil {
		return "", errx.New(err)
	}

	path := conf.DynamicReportPrefix + fileName
	presignedName := reports.GetFileNameNoUUID(fileName)

	return c.s3Client.PreSignURL(conf.ReportBucket, path, presignedName, urlExpireMinutes)
}

func (c *modsImpl) GetFittedDateRange(ctx context.Context, modID, reportType, startDate, endDate string) (string, string, error) {
	dbStartDate, dbEndDate, err := c.piperdbclient.GetDomainReportDateRange(ctx, backendDomain, reportType, modID)
	if err != nil {
		return "", "", err
	}

	return reports.GetFittedDateRange(startDate, endDate, dbStartDate, dbEndDate)
}
