package mods

import (
	"strconv"
	"testing"
	"time"

	"code.justin.tv/insights/piper-service/backend/mocks"
	s3reportmock "code.justin.tv/insights/piper-service/backend/s3report/mocks"
	eleriummock "code.justin.tv/insights/piper-service/internal/clients/elerium/mocks"
	piperdbmock "code.justin.tv/insights/piper-service/internal/clients/piperdb/mocks"
	s3mock "code.justin.tv/insights/piper-service/internal/clients/s3/mocks"
	spademock "code.justin.tv/insights/piper-service/internal/clients/spade/mocks"
	usersmock "code.justin.tv/insights/piper-service/internal/clients/users/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"

	piperconfig "code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/models"
)

const (
	reportType   = "overview_v1"
	nonModReport = "not_this_type"
	dateLayout   = "2006-01-02"
)

type GetModDownloadURLTest struct {
	suite.Suite

	mockPiperDB  *piperdbmock.Client
	mockS3Report *s3reportmock.Backend
	mockS3       *s3mock.Client
	mockElerium  *eleriummock.Client
	mockUsers    *usersmock.Client
	mockSpade    *spademock.Client
	mockCacher   *mocks.Cacher
	startDate    time.Time
	endDate      time.Time

	backend     Backend
	piperconfig piperconfig.PiperConfig
	ctx         context.Context
}

func (suite *GetModDownloadURLTest) SetupTest() {
	var err error

	suite.mockPiperDB = &piperdbmock.Client{}
	suite.mockS3Report = &s3reportmock.Backend{}
	suite.mockS3 = &s3mock.Client{}
	suite.mockElerium = &eleriummock.Client{}
	suite.mockUsers = &usersmock.Client{}
	suite.mockCacher = &mocks.Cacher{}

	suite.startDate = time.Now().AddDate(0, 2, 0).UTC()
	suite.endDate = time.Now().AddDate(0, 1, 0).UTC()

	suite.backend, err = NewBackend(suite.mockPiperDB, suite.mockS3Report, suite.mockS3, suite.mockElerium, suite.mockUsers, suite.mockSpade, suite.mockCacher)
	assert.NoError(suite.T(), err)
	suite.piperconfig = piperconfig.PiperConfig{ReportBucket: "abucket", DynamicReportPrefix: "aprefix/"}
	suite.ctx = context.Background()
}

func (suite *GetModDownloadURLTest) TearDownTest() {
	suite.mockCacher.AssertExpectations(suite.T())
	suite.mockUsers.AssertExpectations(suite.T())
	suite.mockS3Report.AssertExpectations(suite.T())
	suite.mockPiperDB.AssertExpectations(suite.T())
	suite.mockElerium.AssertExpectations(suite.T())
	suite.mockS3.AssertExpectations(suite.T())
}

func (suite *GetModDownloadURLTest) TestWrongtReportType() {
	url, err := suite.backend.GetModReportURL(suite.ctx, staffUser, "111", nonModReport, suite.piperconfig)
	assert.Equal(suite.T(), "", url)
	assert.Error(suite.T(), models.ErrInvalidReportType, err.Error())
}

func (suite *GetModDownloadURLTest) TestNoUserAccess() {
	modResp := make([]models.Mod, 2)
	modResp = append(modResp, models.Mod{Id: 111, Name: "test mod no.1"})
	modResp = append(modResp, models.Mod{Id: 222, Name: "test mod no.2"})

	suite.mockUsers.On("IsStaff", suite.ctx, nonStaffUserNotOwnsMod).Return(false, nil)
	suite.mockElerium.On("GetModsByUserID", suite.ctx, nonStaffUserNotOwnsMod, suite.piperconfig).Return(&modResp, nil)

	url, err := suite.backend.GetModReportURL(suite.ctx, nonStaffUserNotOwnsMod, "333", reportType, suite.piperconfig)
	assert.Equal(suite.T(), "", url)
	assert.Error(suite.T(), models.ErrAccessForbidden, err.Error())
}

func (suite *GetModDownloadURLTest) TestGetModNameFromDB() {
	modResp := make([]models.Mod, 2)
	modResp = append(modResp, models.Mod{Id: 111, Name: "test mod no.1"})
	modResp = append(modResp, models.Mod{Id: 222, Name: "test mod no.2"})

	userRequestMod := models.Mod{Id: 333, Name: "mod no.3"}

	startStr := suite.startDate.Format(dateLayout)
	endStr := suite.endDate.Format(dateLayout)
	suite.mockCacher.On("GetStringProperties", suite.ctx, getModNamePrefix+getModNameDelimeter+strconv.Itoa(userRequestMod.Id)).Return(userRequestMod.Name, true)
	suite.mockS3.On("PreSignURL", "abucket", suite.piperconfig.DynamicReportPrefix+"afilename", ".csv", 5).Return("", nil)
	suite.mockS3Report.On("SaveReportToS3", suite.ctx, [][]string{}, userRequestMod.Name, reportType, startStr, endStr, suite.piperconfig.ReportBucket, suite.piperconfig.DynamicReportPrefix).Return("afilename", nil)
	suite.mockPiperDB.On("GetModReportByDate", suite.ctx, strconv.Itoa(userRequestMod.Id), reportType, startStr, endStr).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetDomainReportDateRange", suite.ctx, backendDomain, reportType, strconv.Itoa(userRequestMod.Id)).Return(suite.startDate, suite.endDate, nil)
	suite.mockUsers.On("IsStaff", suite.ctx, staffUser).Return(true, nil)
	suite.mockElerium.On("GetModsByUserID", suite.ctx, staffUser, suite.piperconfig).Return(&modResp, nil)

	url, err := suite.backend.GetModReportURL(suite.ctx, staffUser, strconv.Itoa(userRequestMod.Id), reportType, suite.piperconfig)
	assert.Equal(suite.T(), "", url)
	assert.NoError(suite.T(), err)
}

func (suite *GetModDownloadURLTest) TestGetNoModNameFromDB() {
	modResp := make([]models.Mod, 2)
	modResp = append(modResp, models.Mod{Id: 111, Name: "test mod no.1"})
	modResp = append(modResp, models.Mod{Id: 222, Name: "test mod no.2"})

	userRequestMod := models.Mod{Id: 333, Name: ""}

	startStr := suite.startDate.Format(dateLayout)
	endStr := suite.endDate.Format(dateLayout)

	suite.mockCacher.On("GetStringProperties", suite.ctx, getModNamePrefix+getModNameDelimeter+strconv.Itoa(userRequestMod.Id)).Return(userRequestMod.Name, true)

	suite.mockS3.On("PreSignURL", "abucket", suite.piperconfig.DynamicReportPrefix+"afilename", ".csv", 5).Return("", nil)
	suite.mockS3Report.On("SaveReportToS3", suite.ctx, [][]string{}, userRequestMod.Name, reportType, startStr, endStr, suite.piperconfig.ReportBucket, suite.piperconfig.DynamicReportPrefix).Return("afilename", nil)
	suite.mockPiperDB.On("GetModReportByDate", suite.ctx, strconv.Itoa(userRequestMod.Id), reportType, startStr, endStr).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetDomainReportDateRange", suite.ctx, backendDomain, reportType, strconv.Itoa(userRequestMod.Id)).Return(suite.startDate, suite.endDate, nil)
	suite.mockUsers.On("IsStaff", suite.ctx, staffUser).Return(true, nil)
	suite.mockElerium.On("GetModsByUserID", suite.ctx, staffUser, suite.piperconfig).Return(&modResp, nil)

	url, err := suite.backend.GetModReportURL(suite.ctx, staffUser, strconv.Itoa(userRequestMod.Id), reportType, suite.piperconfig)
	assert.Equal(suite.T(), "", url)
	assert.NoError(suite.T(), err)
}

func (suite *GetModDownloadURLTest) TestAllCorrect() {
	modResp := make([]models.Mod, 2)
	modResp = append(modResp, models.Mod{Id: 111, Name: "test mod no.1"})
	modResp = append(modResp, models.Mod{Id: 222, Name: "test mod no.2"})

	userRequestMod := models.Mod{Id: 111, Name: "test mod no.1"}

	startStr := suite.startDate.Format(dateLayout)
	endStr := suite.endDate.Format(dateLayout)

	suite.mockS3.On("PreSignURL", "abucket", suite.piperconfig.DynamicReportPrefix+"afilename", ".csv", 5).Return("", nil)
	suite.mockS3Report.On("SaveReportToS3", suite.ctx, [][]string{}, userRequestMod.Name, reportType, startStr, endStr, suite.piperconfig.ReportBucket, suite.piperconfig.DynamicReportPrefix).Return("afilename", nil)
	suite.mockPiperDB.On("GetModReportByDate", suite.ctx, strconv.Itoa(userRequestMod.Id), reportType, startStr, endStr).Return([][]string{}, nil)
	suite.mockPiperDB.On("GetDomainReportDateRange", suite.ctx, backendDomain, reportType, strconv.Itoa(userRequestMod.Id)).Return(suite.startDate, suite.endDate, nil)
	suite.mockElerium.On("GetModsByUserID", suite.ctx, staffUser, suite.piperconfig).Return(&modResp, nil)

	url, err := suite.backend.GetModReportURL(suite.ctx, staffUser, strconv.Itoa(userRequestMod.Id), reportType, suite.piperconfig)
	assert.Equal(suite.T(), "", url)
	assert.NoError(suite.T(), err)
}

func TestGetModDownloadURLTest(t *testing.T) {
	suite.Run(t, new(GetModDownloadURLTest))
}
