package mods

import (
	"testing"
	"time"

	"code.justin.tv/insights/piper-service/lib/reports"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"

	"code.justin.tv/insights/piper-service/backend/mocks"
	s3reportmock "code.justin.tv/insights/piper-service/backend/s3report/mocks"
	eleriummock "code.justin.tv/insights/piper-service/internal/clients/elerium/mocks"
	piperdbmock "code.justin.tv/insights/piper-service/internal/clients/piperdb/mocks"
	s3mock "code.justin.tv/insights/piper-service/internal/clients/s3/mocks"
	spademock "code.justin.tv/insights/piper-service/internal/clients/spade/mocks"
	usersmock "code.justin.tv/insights/piper-service/internal/clients/users/mocks"
	piperconfig "code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/models"
)

const (
	staffUser              = "11111"
	nonStaffUserOwnsMod    = "22222"
	nonStaffUserNotOwnsMod = "33333"
)

type GetModsTest struct {
	suite.Suite
	mockS3       *s3mock.Client
	mockPiperDB  *piperdbmock.Client
	mockS3Report *s3reportmock.Backend
	mockElerium  *eleriummock.Client
	mockUsers    *usersmock.Client
	mockSpade    *spademock.Client
	mockCacher   *mocks.Cacher
	startDate    time.Time
	endDate      time.Time

	backend     Backend
	piperconfig piperconfig.PiperConfig
	ctx         context.Context
}

func (suite *GetModsTest) SetupTest() {
	var err error

	suite.mockS3 = &s3mock.Client{}
	suite.mockPiperDB = &piperdbmock.Client{}
	suite.mockS3Report = &s3reportmock.Backend{}
	suite.mockElerium = &eleriummock.Client{}
	suite.mockUsers = &usersmock.Client{}
	suite.mockCacher = &mocks.Cacher{}

	suite.startDate = time.Date(2018, time.Month(10), 1, 0, 0, 0, 0, time.UTC)
	suite.endDate = time.Date(2018, time.Month(11), 1, 0, 0, 0, 0, time.UTC)

	suite.backend, err = NewBackend(suite.mockPiperDB, suite.mockS3Report, suite.mockS3, suite.mockElerium, suite.mockUsers, suite.mockSpade, suite.mockCacher)
	assert.NoError(suite.T(), err)
	suite.piperconfig = piperconfig.PiperConfig{}
	suite.ctx = context.Background()
}

func (suite *GetModsTest) TearDownTest() {
	suite.mockCacher.AssertExpectations(suite.T())
	suite.mockUsers.AssertExpectations(suite.T())
	suite.mockS3Report.AssertExpectations(suite.T())
	suite.mockPiperDB.AssertExpectations(suite.T())
	suite.mockElerium.AssertExpectations(suite.T())
	suite.mockS3.AssertExpectations(suite.T())
}

func (suite *GetModsTest) TestGetModsByUserIDUserOwnsMods() {
	var modResp []models.Mod
	modResp = append(modResp, models.Mod{Id: 111, Name: "test mod no.1"})
	modResp = append(modResp, models.Mod{Id: 222, Name: "test mod no.2"})

	modIDs := []string{"111", "222"}
	var expected models.Reports
	expected = append(expected, models.Report{ID: "111", ReportType: reports.DevsiteModsReportType})
	expected = append(expected, models.Report{ID: "222", ReportType: reports.DevsiteModsReportType})

	suite.mockPiperDB.On("GetAvailableDomainsWithReports", suite.ctx, backendDomain, reports.DevsiteModsReportType, modIDs).Return(modIDs, nil)
	suite.mockElerium.On("GetModsByUserID", suite.ctx, nonStaffUserOwnsMod, suite.piperconfig).Return(&modResp, nil)
	result, err := suite.backend.GetModReports(suite.ctx, nonStaffUserOwnsMod, suite.piperconfig)
	require.NoError(suite.T(), err)
	assert.Equal(suite.T(), &expected, result)
}

func (suite *GetModsTest) TestUserCanAccessUserOwnsMods() {
	modResp := make([]models.Mod, 2)
	mod1 := models.Mod{Id: 111, Name: "test mod no.1"}
	mod2 := models.Mod{Id: 222, Name: "test mod no.2"}
	modResp = append(modResp, mod1)
	modResp = append(modResp, mod2)

	suite.mockElerium.On("GetModsByUserID", suite.ctx, nonStaffUserOwnsMod, suite.piperconfig).Return(&modResp, nil)
	flag, result, err := suite.backend.UserCanAccess(suite.ctx, nonStaffUserOwnsMod, "222", suite.piperconfig)

	assert.Equal(suite.T(), true, flag)
	assert.Equal(suite.T(), &mod2, result)
	assert.NoError(suite.T(), err)
}

func (suite *GetModsTest) TestUserCanAccessNonStaffUserNoMods() {
	modResp := make([]models.Mod, 2)
	modResp = append(modResp, models.Mod{Id: 111, Name: "test mod no.1"})
	modResp = append(modResp, models.Mod{Id: 222, Name: "test mod no.2"})

	var expMod *models.Mod

	suite.mockUsers.On("IsStaff", suite.ctx, nonStaffUserNotOwnsMod).Return(false, nil)
	suite.mockElerium.On("GetModsByUserID", suite.ctx, nonStaffUserNotOwnsMod, suite.piperconfig).Return(&modResp, nil)
	flag, result, err := suite.backend.UserCanAccess(suite.ctx, nonStaffUserNotOwnsMod, "333", suite.piperconfig)
	assert.Equal(suite.T(), false, flag)
	assert.Equal(suite.T(), expMod, result)
	assert.NoError(suite.T(), err)
}

func (suite *GetModsTest) TestUserCanAccessStaffUser() {
	modResp := make([]models.Mod, 2)
	modResp = append(modResp, models.Mod{Id: 111, Name: "test mod no.1"})
	modResp = append(modResp, models.Mod{Id: 222, Name: "test mod no.2"})

	suite.mockUsers.On("IsStaff", suite.ctx, staffUser).Return(true, nil)
	suite.mockElerium.On("GetModsByUserID", suite.ctx, staffUser, suite.piperconfig).Return(&modResp, nil)
	flag, result, err := suite.backend.UserCanAccess(suite.ctx, staffUser, "333", suite.piperconfig)

	assert.Equal(suite.T(), true, flag)
	assert.Equal(suite.T(), (*models.Mod)(nil), result)
	assert.NoError(suite.T(), err)
}

func (suite *GetModsTest) TestModHasReportsNoRow() {
	var modResp []models.Mod
	modResp = append(modResp, models.Mod{Id: 111, Name: "test mod no.1"})
	modResp = append(modResp, models.Mod{Id: 222, Name: "test mod no.2"})

	modIDs := []string{"111", "222"}
	var expected models.Reports

	suite.mockPiperDB.On("GetAvailableDomainsWithReports", suite.ctx, backendDomain, reports.DevsiteModsReportType, modIDs).Return([]string{}, nil)
	suite.mockElerium.On("GetModsByUserID", suite.ctx, nonStaffUserOwnsMod, suite.piperconfig).Return(&modResp, nil)
	result, err := suite.backend.GetModReports(suite.ctx, nonStaffUserOwnsMod, suite.piperconfig)
	require.NoError(suite.T(), err)
	assert.Equal(suite.T(), &expected, result)
}

func TestGetModsTest(t *testing.T) {
	suite.Run(t, new(GetModsTest))
}
