package one_pager

import (
	"time"

	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/insights/piper-service/backend"
	"code.justin.tv/insights/piper-service/backend/s3report"
	"code.justin.tv/insights/piper-service/internal/clients/discovery"
	"code.justin.tv/insights/piper-service/internal/clients/ems"
	"code.justin.tv/insights/piper-service/internal/clients/piperdb"
	"code.justin.tv/insights/piper-service/internal/clients/rbac"
	"code.justin.tv/insights/piper-service/internal/clients/s3"
	"code.justin.tv/insights/piper-service/internal/clients/users"
	"code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

//go:generate mockery -name Backend
type Backend interface {
	GetGameOnePager(ctx context.Context, gameID string, yearStr, quarterStr string) (models.GameOnePagerDataResponse, error)
	UserCanAccess(ctx context.Context, userID, gameID, companyID string) (bool, error)
	TrackOnePagerUsage(ctx context.Context, gamesEvent models.SpadeEvent)
	AddProfileImageToResp(ctx context.Context, resp *models.GameOnePagerDataResponse) error
	GetGameOnePagerDownloadURL(ctx context.Context, gameID string, yearStr, quarterStr string, conf config.PiperConfig) (interface{}, string, error)
	ConvertYearQuarterToStartDate(ctx context.Context, yearStr, quarterStr string) (time.Time, error)
	GetQuarterStartRange(ctx context.Context, gameID string) (models.QuarterStartRange, error)
	GetDefaultQuarterStartRange(ctx context.Context) (models.QuarterStartRange, error)
}

type onePagerImpl struct {
	piperDBClient   piperdb.Client
	discoveryClient discovery.Client
	spadeClient     spade.Client
	rbacClient      rbac.Client
	usersClient     users.Client
	emsClient       ems.Client
	s3Client        s3.Client
	s3report        s3report.Backend
	cacher          backend.Cacher
}

func NewBackend(piperDBClient piperdb.Client, discoveryClient discovery.Client, spadeClient spade.Client, rbacClient rbac.Client, usersClient users.Client, emsClient ems.Client, s3Client s3.Client, s3report s3report.Backend, cacher backend.Cacher) (Backend, error) {
	return &onePagerImpl{
		piperDBClient,
		discoveryClient,
		spadeClient,
		rbacClient,
		usersClient,
		emsClient,
		s3Client,
		s3report,
		cacher,
	}, nil
}
