package one_pager

import (
	"fmt"
	"strings"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

func (o onePagerImpl) transferDefaultDataToGameDataResponse(ctx context.Context, defaultGameData models.OnePagerDefaultData, status models.OnePagerSectionPresentStatus) (models.GameOnePagerDataResponse, error) {
	defaultGameResp := models.GameOnePagerDataResponse{}

	if !status.OverviewStats {
		defaultGameResp.GeneralStats = &models.TwitchOverviewStats{
			MinutesWatched:            defaultGameData.MinutesWatched,
			UniqueMonthlyBroadcasters: defaultGameData.UniqueMonthlyBroadcasters,
			UniqueDailyVisitors:       defaultGameData.UniqueDailyVisitors,
			PartneredStreamers:        defaultGameData.PartneredStreamers,
			AffiliateStreamers:        defaultGameData.AffiliateStreamers,
			TotalClips:                defaultGameData.TotalClips,
		}
	}

	if !status.BroadcasterStats {
		var popularBroadcasters []models.PopularBroadcaster
		broadcasterInfos := strings.Split(defaultGameData.TopThreeBroadcaster, itemDelimiter)
		for _, info := range broadcasterInfos {
			loginAndDisplayName := strings.Split(info, sectionDelimiter)
			if len(loginAndDisplayName) != 2 {
				logx.Error(ctx, fmt.Sprintf("Can't parse popular broadcasters for db string %s", info))
				return defaultGameResp, models.ErrBadGameInsightsData
			}

			popularBroadcasters = append(popularBroadcasters, models.PopularBroadcaster{
				Name:        loginAndDisplayName[0],
				DisplayName: loginAndDisplayName[1],
			})
		}
		defaultGameResp.PopularBroadcasters = &popularBroadcasters
	}

	if !status.ExtensionStats {
		extensions := o.parseExtensions(ctx, defaultGameData.TopThreeExtensions, nil)
		// return error when no extensions are actually parsed out or no metadata are correctly set
		if len(extensions) == 0 {
			return defaultGameResp, models.ErrBadGameInsightsData
		}
		defaultGameResp.PopularExtensions = &extensions
	}

	if !status.AudienceStats {
		popularGamesByStreamers := o.parseGame(ctx, defaultGameData.TopThreeGamesByStreamers)

		// return error when no games are actually parsed out or no metadata are correctly set
		if len(popularGamesByStreamers) == 0 {
			return defaultGameResp, models.ErrBadGameInsightsData
		}

		// return error when no games are actually parsed out or no metadata are correctly set
		popularGamesByViewers := o.parseGame(ctx, defaultGameData.TopThreeGamesByViewers)
		if len(popularGamesByViewers) == 0 {
			return defaultGameResp, models.ErrBadGameInsightsData
		}

		defaultGameResp.PopularGames = &models.PopularGames{
			PopularGamesToWatch:  transferToPopularGames(popularGamesByViewers),
			PopularGamesToStream: transferToPopularGames(popularGamesByStreamers),
		}
	}

	defaultGameResp.NumOfExtensions = defaultGameData.TotalActiveExtensions

	return defaultGameResp, nil
}

// fill in all the data from default if it is missing from database
func combineDefaultDataAndGameData(defaultGameData, gameData models.GameOnePagerDataResponse, status models.OnePagerSectionPresentStatus) (models.GameOnePagerDataResponse, error) {
	gameDataResp := models.GameOnePagerDataResponse{
		GameInfo: gameData.GameInfo,
	}

	if gameData.NumOfExtensions != 0 {
		gameDataResp.NumOfExtensions = gameData.NumOfExtensions
	} else if defaultGameData.NumOfExtensions != 0 {
		gameDataResp.NumOfExtensions = defaultGameData.NumOfExtensions
	} else {
		return gameDataResp, errx.New(models.ErrBadGameInsightsData)
	}

	if status.OverviewStats && gameData.OverviewStats != nil {
		gameDataResp.OverviewStats = gameData.OverviewStats
	} else if defaultGameData.GeneralStats != nil {
		gameDataResp.GeneralStats = defaultGameData.GeneralStats
	} else {
		return gameDataResp, errx.New(models.ErrBadGameInsightsData)
	}

	if status.BroadcasterStats && gameData.BroadcasterStats != nil {
		gameDataResp.BroadcasterStats = gameData.BroadcasterStats
	} else if defaultGameData.PopularBroadcasters != nil {
		gameDataResp.PopularBroadcasters = defaultGameData.PopularBroadcasters
	} else {
		return gameDataResp, errx.New(models.ErrBadGameInsightsData)
	}

	if status.SocialStats && gameData.SocialStats != nil {
		gameDataResp.SocialStats = gameData.SocialStats
	} else {
		gameDataResp.SocialStats = nil
	}

	if status.ExtensionStats && gameData.ExtensionStats != nil {
		gameDataResp.ExtensionStats = gameData.ExtensionStats
	} else if defaultGameData.PopularExtensions != nil {
		gameDataResp.PopularExtensions = defaultGameData.PopularExtensions
	} else {
		return gameDataResp, errx.New(models.ErrBadGameInsightsData)
	}

	if status.DropStats && gameData.DropsStats != nil {
		gameDataResp.DropsStats = gameData.DropsStats
	} else {
		gameDataResp.DropsStats = nil
	}

	if status.AudienceStats && gameData.AudienceStats != nil {
		gameDataResp.AudienceStats = gameData.AudienceStats
	} else if defaultGameData.PopularGames != nil {
		gameDataResp.PopularGames = defaultGameData.PopularGames
	} else {
		return gameDataResp, errx.New(models.ErrBadGameInsightsData)
	}

	return gameDataResp, nil
}

func transferToPopularGames(topGames []models.TopGame) []models.TwitchPopularGame {
	var games []models.TwitchPopularGame
	for _, topGame := range topGames {
		games = append(games, models.TwitchPopularGame{
			Id:   topGame.Id,
			Name: topGame.Name,
			Img:  topGame.Img,
		})
	}
	return games
}
