package one_pager

import (
	"testing"

	"code.justin.tv/insights/piper-service/internal/utils"
	"code.justin.tv/insights/piper-service/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/stretchr/testify/suite"
)

type GetDefaultInsightsTest struct {
	onePagerSuite
}

func (suite *GetDefaultInsightsTest) TestTransferDefaultDataToGameDataResponseAllMissing() {
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, "2b9s9g75awk0ju391l0o2w4e7jxo47").Return(models.ExtensionStats{Id: "2b9s9g75awk0ju391l0o2w4e7jxo47", Name: "popularExtension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, "gh70y1spw727ohtgzbhc0hppvq9br2").Return(models.ExtensionStats{Id: "gh70y1spw727ohtgzbhc0hppvq9br2", Name: "popularExtension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, "wi08ebtatdc7oj83wtl9uxwz807l8b").Return(models.ExtensionStats{Id: "wi08ebtatdc7oj83wtl9uxwz807l8b", Name: "popularExtension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "1458").Return(models.TopGame{Id: "1458", Name: "popularGameName1", Percentage: 45, Img: "popularGameImg1"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4609").Return(models.TopGame{Id: "4609", Name: "popularGameName2", Percentage: 28.1, Img: "popularGameImg2"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "3734").Return(models.TopGame{Id: "3734", Name: "popularGameName3", Percentage: 10.28, Img: "popularGameImg3"}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "46091").Return(models.TopGame{Id: "46091", Name: "popularGameName4", Percentage: 38.12, Img: "popularGameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "582").Return(models.TopGame{Id: "582", Name: "popularGameName5", Percentage: 32.19, Img: "popularGameImg5"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "37341").Return(models.TopGame{Id: "37341", Name: "popularGameName6", Percentage: 12.28, Img: "popularGameImg6"}, nil)

	resp, err := suite.onePagerImpl.transferDefaultDataToGameDataResponse(suite.ctx, onePagerDefaultData, models.OnePagerSectionPresentStatus{
		OverviewStats:    false,
		BroadcasterStats: false,
		SocialStats:      false,
		ExtensionStats:   false,
		DropStats:        false,
		AudienceStats:    false,
	})
	suite.NoError(err)

	if diff := utils.Equal(resp, onePagerDefaultDataResp); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
}

func (suite *GetDefaultInsightsTest) TestCombineDefaultAndActualAllPresent() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: true,
		SocialStats:      true,
		ExtensionStats:   true,
		DropStats:        true,
		AudienceStats:    true,
	}

	resp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, false)
	suite.NoError(err)

	finalResp, err := combineDefaultDataAndGameData(onePagerDefaultDataResp, resp, presence)
	suite.NoError(err)

	if diff := utils.Equal(resp, finalResp); diff != nil {
		suite.T().Error(diff)
	}
}

func (suite *GetDefaultInsightsTest) TestCombineDefaultAndActualMissingOverviewStats() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    false,
		BroadcasterStats: true,
		SocialStats:      true,
		ExtensionStats:   true,
		DropStats:        true,
		AudienceStats:    true,
	}

	resp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, false)
	suite.NoError(err)

	finalResp, err := combineDefaultDataAndGameData(onePagerDefaultDataResp, resp, presence)
	suite.NoError(err)

	if diff := utils.Equal(resp, finalResp); diff != nil {
		suite.T().Error(diff)
	}
}

func (suite *GetDefaultInsightsTest) TestCombineDefaultAndActualMissingBroadcasterStats() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: false,
		SocialStats:      true,
		ExtensionStats:   true,
		DropStats:        true,
		AudienceStats:    true,
	}

	resp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, false)
	suite.NoError(err)

	finalResp, err := combineDefaultDataAndGameData(onePagerDefaultDataResp, resp, presence)
	suite.NoError(err)

	if diff := utils.Equal(resp, finalResp); diff != nil {
		suite.T().Error(diff)
	}
}

func (suite *GetDefaultInsightsTest) TestCombineDefaultAndActualMissingSocialStats() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: true,
		SocialStats:      false,
		ExtensionStats:   true,
		DropStats:        true,
		AudienceStats:    true,
	}

	resp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, false)
	suite.NoError(err)

	finalResp, err := combineDefaultDataAndGameData(onePagerDefaultDataResp, resp, presence)
	suite.NoError(err)

	if diff := utils.Equal(resp, finalResp); diff != nil {
		suite.T().Error(diff)
	}
}

func (suite *GetDefaultInsightsTest) TestCombineDefaultAndActualMissingExtensionStats() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: true,
		SocialStats:      true,
		ExtensionStats:   false,
		DropStats:        true,
		AudienceStats:    true,
	}

	resp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, false)
	suite.NoError(err)

	finalResp, err := combineDefaultDataAndGameData(onePagerDefaultDataResp, resp, presence)
	suite.NoError(err)

	if diff := utils.Equal(resp, finalResp); diff != nil {
		suite.T().Error(diff)
	}
}

func (suite *GetDefaultInsightsTest) TestCombineDefaultAndActualMissingDropStats() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: true,
		SocialStats:      true,
		ExtensionStats:   true,
		DropStats:        false,
		AudienceStats:    true,
	}

	resp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, false)
	suite.NoError(err)

	finalResp, err := combineDefaultDataAndGameData(onePagerDefaultDataResp, resp, presence)
	suite.NoError(err)

	if diff := utils.Equal(resp, finalResp); diff != nil {
		suite.T().Error(diff)
	}
}

func (suite *GetDefaultInsightsTest) TestCombineDefaultAndActualMissingAudienceStats() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: true,
		SocialStats:      true,
		ExtensionStats:   true,
		DropStats:        true,
		AudienceStats:    false,
	}

	resp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, false)
	suite.NoError(err)

	finalResp, err := combineDefaultDataAndGameData(onePagerDefaultDataResp, resp, presence)
	suite.NoError(err)

	if diff := utils.Equal(resp, finalResp); diff != nil {
		suite.T().Error(diff)
	}
}

func TestGetDefaultInsights(t *testing.T) {
	suite.Run(t, new(GetDefaultInsightsTest))
}
