package one_pager

import (
	"fmt"
	"time"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/lib/reports"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

const (
	reportName       = "one_pager"
	urlExpireMinutes = 5
)

func (o onePagerImpl) GetGameOnePagerDownloadURL(ctx context.Context, gameID string, yearStr, quarterStr string, conf config.PiperConfig) (interface{}, string, error) {
	startDate, err := o.ConvertYearQuarterToStartDate(ctx, yearStr, quarterStr)
	if err != nil {
		return "", "", err
	}

	status, _, resp, err := o.gatherCustomGameData(ctx, gameID, startDate, yearStr, quarterStr)

	if err != nil {
		return "", "", err
	}

	if !status.OverviewStats && !status.BroadcasterStats && !status.SocialStats && !status.ExtensionStats && !status.DropStats && !status.AudienceStats {
		return "", "", models.ErrReportNotFound
	}

	report := flattenGameOnePagerDataResponse(resp, &status, startDate)

	filename, err := o.saveReportToS3(ctx, report, resp.GameInfo.GameName, reportName, yearStr, quarterStr, conf)
	if err != nil {
		return "", "", errx.New(err)
	}

	path := conf.DynamicReportPrefix + filename
	preSignedName := reports.GetFileNameNoUUID(filename)
	preSigned, err := o.preSignURL(conf.ReportBucket, path, preSignedName)
	if err != nil {
		return "", "", err
	}
	wrapped := wrapResults("url", preSigned)

	return wrapped, resp.GameInfo.GameName, nil
}

func flattenGameOnePagerDataResponse(resp models.GameOnePagerDataResponse, status *models.OnePagerSectionPresentStatus, startDate time.Time) [][]string {
	header := make([]string, 0)
	content := make([]string, 0)
	report := make([][]string, 2)

	setupQuarterStart(startDate, &header, &content)

	setupGamesInfo(resp, &header, &content)

	if !ifStatsMissing(status.OverviewStats) && !ifValueIsNil(resp.OverviewStats) {
		setupOverviewStats(resp, &header, &content)
	}

	if !ifStatsMissing(status.BroadcasterStats) && !ifValueIsNil(resp.BroadcasterStats) {
		setupBroadcasterStats(resp, &header, &content)
	}

	if !ifStatsMissing(status.SocialStats) && !ifValueIsNil(resp.SocialStats) {
		setupSocialStats(resp, &header, &content)
	}

	if !ifStatsMissing(status.AudienceStats) && !ifValueIsNil(resp.AudienceStats) {
		setupAudienceStats(resp, &header, &content)
	}

	setupNumOfExtensions(resp, &header, &content)

	if !ifStatsMissing(status.ExtensionStats) && !ifValueIsNil(resp.ExtensionStats) {
		setupExtensionStats(resp, &header, &content)
	}

	if !ifStatsMissing(status.DropStats) && !ifValueIsNil(resp.DropsStats) {
		setupDropStats(resp, &header, &content)
	}

	report[0] = header
	report[1] = content

	return report
}

func setupQuarterStart(startDate time.Time, header *[]string, content *[]string) {
	appendHeader(header, "Quarter Start")
	appendContent(content, convertTimeToStr(startDate))
}

func setupGamesInfo(resp models.GameOnePagerDataResponse, header *[]string, content *[]string) {
	if resp.GameInfo.GameID != "" {
		appendHeader(header, "Game ID")
		appendContent(content, resp.GameInfo.GameID)
	}

	if resp.GameInfo.GameName != "" {
		appendHeader(header, "Game Name")
		appendContent(content, resp.GameInfo.GameName)
	}
}

func setupNumOfExtensions(resp models.GameOnePagerDataResponse, header *[]string, content *[]string) {
	if resp.NumOfExtensions != missingDataNum {
		appendHeader(header, "Total Number Of Extensions on Twitch")
		appendContent(content, convertIntToStr(resp.NumOfExtensions))
	}
}

func setupOverviewStats(resp models.GameOnePagerDataResponse, header *[]string, content *[]string) {
	if resp.OverviewStats.HoursWatched != missingDataNum {
		appendHeader(header, "Hours Watched")
		appendContent(content, convertFloatToStr(resp.OverviewStats.HoursWatched))
	}

	if resp.OverviewStats.HoursWatchedChanges != missingDataNum {
		appendHeader(header, "Hours Watched Q/Q Change	")
		appendContent(content, convertFloatToStr(resp.OverviewStats.HoursWatchedChanges))
	}

	if resp.OverviewStats.UniqueViewers != missingDataNum {
		appendHeader(header, "Unique Viewers")
		appendContent(content, convertIntToStr(resp.OverviewStats.UniqueViewers))
	}

	if resp.OverviewStats.UniqueViewersChanges != missingDataNum {
		appendHeader(header, "Unique Viewers Q/Q Change")
		appendContent(content, convertFloatToStr(resp.OverviewStats.UniqueViewersChanges))
	}

	if resp.OverviewStats.ViewingSessions != missingDataNum {
		appendHeader(header, "Viewing Sessions")
		appendContent(content, convertIntToStr(resp.OverviewStats.ViewingSessions))
	}

	if resp.OverviewStats.ViewingSessionsChanges != missingDataNum {
		appendHeader(header, "Viewing Sessions Q/Q Change")
		appendContent(content, convertFloatToStr(resp.OverviewStats.ViewingSessionsChanges))
	}

	if resp.OverviewStats.HoursBroadcast != missingDataNum {
		appendHeader(header, "Hours Broadcast")
		appendContent(content, convertFloatToStr(resp.OverviewStats.HoursBroadcast))
	}

	if resp.OverviewStats.HoursBroadcastChanges != missingDataNum {
		appendHeader(header, "Hours Broadcast Q/Q Change")
		appendContent(content, convertFloatToStr(resp.OverviewStats.HoursBroadcastChanges))
	}

	if resp.OverviewStats.UniqueBroadcasters != missingDataNum {
		appendHeader(header, "Unique Broadcasters")
		appendContent(content, convertIntToStr(resp.OverviewStats.UniqueBroadcasters))
	}

	if resp.OverviewStats.HoursBroadcastChanges != missingDataNum {
		appendHeader(header, "Unique Broadcasters Q/Q Change")
		appendContent(content, convertFloatToStr(resp.OverviewStats.UniqueBroadcastersChanges))
	}

	if resp.OverviewStats.BroadcastingSessions != missingDataNum {
		appendHeader(header, "Broadcasting Sessions")
		appendContent(content, convertIntToStr(resp.OverviewStats.BroadcastingSessions))
	}

	if resp.OverviewStats.BroadcastingSessionsChanges != missingDataNum {
		appendHeader(header, "Broadcasting Sessions Q/Q Change")
		appendContent(content, convertFloatToStr(resp.OverviewStats.BroadcastingSessionsChanges))
	}
}

func setupBroadcasterStats(resp models.GameOnePagerDataResponse, header *[]string, content *[]string) {
	if resp.BroadcasterStats.TopBroadcaster != nil {
		if resp.BroadcasterStats.TopBroadcaster.Name != missingDataStr {
			appendHeader(header, "Top Broadcaster Name")
			appendContent(content, resp.BroadcasterStats.TopBroadcaster.Name)
		}

		if resp.BroadcasterStats.TopBroadcaster.DisplayName != missingDataStr {
			appendHeader(header, "Top Broadcaster Display Name")
			appendContent(content, resp.BroadcasterStats.TopBroadcaster.DisplayName)
		}

		if resp.BroadcasterStats.TopBroadcaster.HoursBroadcast != missingDataNum {
			appendHeader(header, "Top Broadcaster Hours Broadcast")
			appendContent(content, convertFloatToStr(resp.BroadcasterStats.TopBroadcaster.HoursBroadcast))
		}

		if resp.BroadcasterStats.TopBroadcaster.HoursWatched != missingDataNum {
			appendHeader(header, "Top Broadcaster Hours Watched")
			appendContent(content, convertFloatToStr(resp.BroadcasterStats.TopBroadcaster.HoursWatched))
		}

		if resp.BroadcasterStats.TopBroadcaster.ConcurrentViewers != missingDataNum {
			appendHeader(header, "Top Broadcaster Average Concurrent Viewers")
			appendContent(content, convertIntToStr(resp.BroadcasterStats.TopBroadcaster.ConcurrentViewers))
		}
	}

	if resp.BroadcasterStats.DedicatedBroadcaster != nil {
		if resp.BroadcasterStats.DedicatedBroadcaster.Name != missingDataStr {
			appendHeader(header, "Most Dedicated Broadcaster Name")
			appendContent(content, resp.BroadcasterStats.DedicatedBroadcaster.Name)
		}

		if resp.BroadcasterStats.DedicatedBroadcaster.DisplayName != missingDataStr {
			appendHeader(header, "Most Dedicated Broadcaster Display Name")
			appendContent(content, resp.BroadcasterStats.DedicatedBroadcaster.DisplayName)
		}

		if resp.BroadcasterStats.DedicatedBroadcaster.StreamedSessions != missingDataNum {
			appendHeader(header, "Most Dedicated Broadcaster Streamed Sessions")
			appendContent(content, convertIntToStr(resp.BroadcasterStats.DedicatedBroadcaster.StreamedSessions))
		}

		if resp.BroadcasterStats.DedicatedBroadcaster.AvgStreamSessionsPerWeek != missingDataNum {
			appendHeader(header, "Most Dedicated Broadcaster Hours Watched")
			appendContent(content, convertFloatToStr(resp.BroadcasterStats.DedicatedBroadcaster.AvgStreamSessionsPerWeek))
		}

		if resp.BroadcasterStats.DedicatedBroadcaster.PercentageOfHoursBroadcast != missingDataNum {
			appendHeader(header, "Most Dedicated Broadcaster Percentage Of Hours Broadcast")
			appendContent(content, convertFloatToStr(resp.BroadcasterStats.DedicatedBroadcaster.PercentageOfHoursBroadcast))
		}
	}

	if resp.BroadcasterStats.RisingStar != nil {
		if resp.BroadcasterStats.RisingStar.Name != missingDataStr {
			appendHeader(header, "Rising Star Name")
			appendContent(content, resp.BroadcasterStats.RisingStar.Name)
		}

		if resp.BroadcasterStats.RisingStar.DisplayName != missingDataStr {
			appendHeader(header, "Rising Star Display Name")
			appendContent(content, resp.BroadcasterStats.RisingStar.DisplayName)
		}

		if resp.BroadcasterStats.RisingStar.HoursStreamed != missingDataNum {
			appendHeader(header, "Rising Star Hours Broadcast")
			appendContent(content, convertFloatToStr(resp.BroadcasterStats.RisingStar.HoursStreamed))
		}

		if resp.BroadcasterStats.RisingStar.HoursWatched != missingDataNum {
			appendHeader(header, "Rising Star Hours Watched")
			appendContent(content, convertFloatToStr(resp.BroadcasterStats.RisingStar.HoursWatched))
		}

		if resp.BroadcasterStats.RisingStar.AvgConcurrentViewers != missingDataNum {
			appendHeader(header, "Rising Star Average Concurrent Viewers")
			appendContent(content, convertIntToStr(resp.BroadcasterStats.RisingStar.AvgConcurrentViewers))
		}
	}

}

func setupSocialStats(resp models.GameOnePagerDataResponse, header *[]string, content *[]string) {
	if resp.SocialStats.NumOfChatMessages != missingDataNum {
		appendHeader(header, "Chat Messages")
		appendContent(content, convertIntToStr(resp.SocialStats.NumOfChatMessages))
	}

	if resp.SocialStats.PercentOfEngagement != missingDataNum {
		appendHeader(header, "Audience Engagement Percent")
		appendContent(content, convertFloatToStr(resp.SocialStats.PercentOfEngagement))
	}

	if resp.SocialStats.NumOfPartnersStreamed != missingDataNum {
		appendHeader(header, "Partners and Affiliates Streamed")
		appendContent(content, convertIntToStr(resp.SocialStats.NumOfPartnersStreamed))
	}
}

func setupExtensionStats(resp models.GameOnePagerDataResponse, header *[]string, content *[]string) {
	for i, k := range *resp.ExtensionStats {
		if k.Id != missingDataStr {
			appendHeader(header, fmt.Sprintf("Top %d Extension ID", i+1))
			appendContent(content, k.Id)
		}

		if k.Name != missingDataStr {
			appendHeader(header, fmt.Sprintf("Top %d Extension Name", i+1))
			appendContent(content, k.Name)
		}

		if k.Description != missingDataStr {
			appendHeader(header, fmt.Sprintf("Top %d Extension Description", i+1))
			appendContent(content, k.Description)
		}

		if k.Url != missingDataStr {
			appendHeader(header, fmt.Sprintf("Top %d Extension URL", i+1))
			appendContent(content, k.Url)
		}
	}
}

func setupDropStats(resp models.GameOnePagerDataResponse, header *[]string, content *[]string) {
	if resp.DropsStats.NumOfDropsReceived != missingDataNum {
		appendHeader(header, "Drops Received")
		appendContent(content, convertIntToStr(resp.DropsStats.NumOfDropsReceived))
	}

	if resp.DropsStats.NumOfDropsReceivedChange != missingDataNum {
		appendHeader(header, "Drops Received Q/Q Change")
		appendContent(content, convertFloatToStr(resp.DropsStats.NumOfDropsReceivedChange))
	}

	if resp.DropsStats.NumOfUniqueReceivingUsers != missingDataNum {
		appendHeader(header, "Drops Unique Receiving Users")
		appendContent(content, convertIntToStr(resp.DropsStats.NumOfUniqueReceivingUsers))
	}

	if resp.DropsStats.NumOfUniqueReceivingUsersChanges != missingDataNum {
		appendHeader(header, "Drops Unique Receiving Users Q/Q Change")
		appendContent(content, convertFloatToStr(resp.DropsStats.NumOfUniqueReceivingUsersChanges))
	}

	if resp.DropsStats.NumOfCampaigns != missingDataNum {
		appendHeader(header, "Drops Campaigns")
		appendContent(content, convertIntToStr(resp.DropsStats.NumOfCampaigns))
	}

	if resp.DropsStats.NumOfCampaignsChanges != missingDataNum {
		appendHeader(header, "Drops Campaigns Q/Q Change")
		appendContent(content, convertFloatToStr(resp.DropsStats.NumOfCampaignsChanges))
	}
}

func setupAudienceStats(resp models.GameOnePagerDataResponse, header *[]string, content *[]string) {
	if len(resp.AudienceStats.TopGamesToWatch) > 0 {
		for i, k := range resp.AudienceStats.TopGamesToWatch {
			if k.Id != missingDataStr {
				appendHeader(header, fmt.Sprintf("Top %d Game ID Your Viewers Watched", i+1))
				appendContent(content, k.Id)
			}

			if k.Name != missingDataStr {
				appendHeader(header, fmt.Sprintf("Top %d Game Name Your Viewers Watched", i+1))
				appendContent(content, k.Name)
			}

			if k.Percentage != missingDataNum {
				appendHeader(header, fmt.Sprintf("Top %d Game Percentage Your Viewers Watched", i+1))
				appendContent(content, convertFloatToStr(k.Percentage))
			}
		}
	}

	if len(resp.AudienceStats.TopGamesToStream) > 0 {
		for i, k := range resp.AudienceStats.TopGamesToStream {
			if k.Id != missingDataStr {
				appendHeader(header, fmt.Sprintf("Top %d Game ID Your Streamers Streamed", i+1))
				appendContent(content, k.Id)
			}

			if k.Name != missingDataStr {
				appendHeader(header, fmt.Sprintf("Top %d Game Name Your Streamers Streamed", i+1))
				appendContent(content, k.Name)
			}

			if k.Percentage != missingDataNum {
				appendHeader(header, fmt.Sprintf("Top %d Game Percentage Your Streamers Streamed", i+1))
				appendContent(content, convertFloatToStr(k.Percentage))
			}
		}
	}

	if len(resp.AudienceStats.TopBroadcasterCountries) > 0 {
		for i, k := range resp.AudienceStats.TopBroadcasterCountries {
			if k.Name != missingDataStr {
				appendHeader(header, fmt.Sprintf("Top %d Geo With Broadcaster Growth", i+1))
				appendContent(content, k.Name)
			}

			if k.Percentage != missingDataNum {
				appendHeader(header, fmt.Sprintf("Top %d Geo Growth Percent Broadcaster", i+1))
				appendContent(content, convertFloatToStr(k.Percentage))
			}
		}
	}

	if len(resp.AudienceStats.TopViewerCountries) > 0 {
		for i, k := range resp.AudienceStats.TopViewerCountries {
			if k.Name != missingDataStr {
				appendHeader(header, fmt.Sprintf("Top %d Geo With Viewer Growth", i+1))
				appendContent(content, k.Name)
			}

			if k.Percentage != missingDataNum {
				appendHeader(header, fmt.Sprintf("Top %d Geo Growth Percent Viewer", i+1))
				appendContent(content, convertFloatToStr(k.Percentage))
			}
		}
	}
}

func (o onePagerImpl) preSignURL(bucket string, s3KeyPath string, contentDispositionFileName string) (string, error) {
	url, err := o.s3Client.PreSignURL(bucket, s3KeyPath, contentDispositionFileName, urlExpireMinutes)
	return url, errx.New(err)
}

func (o onePagerImpl) saveReportToS3(ctx context.Context, report [][]string, gameName, reportName string, yearStr, quarterStr string, conf config.PiperConfig) (name string, err error) {
	return o.s3report.SaveReportToS3(ctx, report, gameName, reportName, yearStr, quarterStr, conf.ReportBucket, conf.DynamicReportPrefix)
}
