package one_pager

import (
	"encoding/json"
	"fmt"
	"time"

	"code.justin.tv/chat/golibs/logx"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

const (
	missingDataStr     = "NA"
	missingDataNum     = 0
	onePagerDataPrefix = "op_"
	cacheDelimiter     = "_"
)

func (o onePagerImpl) GetGameOnePager(ctx context.Context, gameID string, yearStr, quarterStr string) (models.GameOnePagerDataResponse, error) {
	gameDataResponse := models.GameOnePagerDataResponse{}
	startDate, err := o.ConvertYearQuarterToStartDate(ctx, yearStr, quarterStr)
	if err != nil {
		return gameDataResponse, err
	}

	status, allPresent, resp, err := o.gatherCustomGameData(ctx, gameID, startDate, yearStr, quarterStr)

	if err != nil {
		return gameDataResponse, err
	}

	// If data is all present, just return here, no need to pull default to fill in missing part
	if allPresent {
		return resp, nil
	}

	// Pull default data from DB to fill in missing part
	onePagerDefaultData, err := o.piperDBClient.GetGameDefaultOnePagerByQuarterAndYear(ctx, startDate)
	if err != nil {
		return gameDataResponse, err
	}

	if _, gameFound := models.BlackListedGames[gameID]; gameFound {
		defaultResp, err := o.transferDefaultDataToGameDataResponse(ctx, onePagerDefaultData, models.OnePagerSectionPresentStatus{
			OverviewStats:    false,
			BroadcasterStats: false,
			SocialStats:      false,
			ExtensionStats:   false,
			DropStats:        false,
			AudienceStats:    false,
		})
		if err != nil {
			return gameDataResponse, err
		}

		gameDataResp := models.GameOnePagerDataResponse{
			GameInfo:            resp.GameInfo,
			NumOfExtensions:     defaultResp.NumOfExtensions,
			GeneralStats:        defaultResp.GeneralStats,
			PopularBroadcasters: defaultResp.PopularBroadcasters,
			SocialStats:         defaultResp.SocialStats,
			PopularExtensions:   defaultResp.PopularExtensions,
			DropsStats:          nil,
			PopularGames:        defaultResp.PopularGames,
		}
		return gameDataResp, nil
	}

	defaultResp, err := o.transferDefaultDataToGameDataResponse(ctx, onePagerDefaultData, status)
	if err != nil {
		return gameDataResponse, err
	}

	gameDataResponse, err = combineDefaultDataAndGameData(defaultResp, resp, status)
	if err != nil {
		return gameDataResponse, err
	}

	return gameDataResponse, nil
}

func (o onePagerImpl) gatherGameMetadata(ctx context.Context, gameID string) (models.GameInfo, error) {
	gameInfo := models.GameInfo{GameID: gameID}

	gameName, err := o.discoveryClient.GetGameNameByID(ctx, gameID)
	if gameName == "" || err != nil {
		return gameInfo, errx.New(models.ErrGameNotFound)
	}

	gameInfo.GameName = gameName

	return gameInfo, nil
}

func (o onePagerImpl) gatherCustomGameData(ctx context.Context, gameID string, startDate time.Time, yearStr, quarterStr string) (models.OnePagerSectionPresentStatus, bool, models.GameOnePagerDataResponse, error) {
	key := fmt.Sprintf("%s%s_%s_%s", onePagerDataPrefix, gameID, yearStr, quarterStr)
	obj, found := o.cacher.GetStringProperties(ctx, key)
	var cachedData models.OnePagerCacheData
	if found {
		if err := json.Unmarshal([]byte(obj), &cachedData); err == nil {
			return cachedData.PresentStatus, cachedData.AllPresent, cachedData.DataResp, nil
		} else {
			logx.Error(ctx, fmt.Sprintf("failed to parse the cached one pager data for %s", key))
		}
	}

	// pull game data
	gameInfo, err := o.gatherGameMetadata(ctx, gameID)
	if err != nil {
		return models.OnePagerSectionPresentStatus{}, false, models.GameOnePagerDataResponse{}, err
	}

	onePagerData, err := o.piperDBClient.GetGameOnePagerByQuarterAndYear(ctx, gameID, startDate)
	if err != nil {
		return models.OnePagerSectionPresentStatus{}, false, models.GameOnePagerDataResponse{}, err
	}

	status, allPresent := checkIfMissingFields(onePagerData)
	resp, err := o.transferToGameDataResponse(ctx, onePagerData, &status)

	// If fetching extension metadata or fetching game metadata are all failing, there is something wrong with the data or server, return error
	if err != nil {
		return models.OnePagerSectionPresentStatus{}, false, models.GameOnePagerDataResponse{}, models.ErrBadGameInsightsData
	}

	resp.GameInfo = gameInfo

	if status.OverviewStats || status.BroadcasterStats || status.SocialStats || status.ExtensionStats || status.DropStats || status.AudienceStats {
		cachedData.PresentStatus, cachedData.DataResp, cachedData.AllPresent = status, resp, allPresent
		cachedDataStr, err := json.Marshal(cachedData)
		if err != nil {
			logx.Error(ctx, fmt.Sprintf("failed to marshall one pager data for %s in cache", key))
		}

		err = o.cacher.CacheStringProperties(ctx, key, string(cachedDataStr))
		if err != nil {
			logx.Error(ctx, fmt.Sprintf("failed to cache one pager data for %s in cache", key))
		}
	}

	return status, allPresent, resp, nil
}
