package one_pager

import (
	"testing"

	"code.justin.tv/insights/piper-service/internal/utils"
	"code.justin.tv/insights/piper-service/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/stretchr/testify/suite"
)

type GetInsightsTest struct {
	onePagerSuite
}

func (suite *GetInsightsTest) SetupTest() {
	suite.onePagerSuite.SetupTest()
}

func (suite *GetInsightsTest) TestDBHasFullData() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: true,
		SocialStats:      true,
		ExtensionStats:   true,
		DropStats:        true,
		AudienceStats:    true,
	}

	dataBaseReturn, err := makeTestOnePagerDataBasedOnMissingStatus(presence)
	suite.NoError(err)

	expectedFullDataResp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, true)
	suite.NoError(err)

	cachedDataStr, err := makeCacheDataString(expectedFullDataResp, presence, true)
	suite.NoError(err)

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension1ID).Return(models.ExtensionStats{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension2ID).Return(models.ExtensionStats{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension3ID).Return(models.ExtensionStats{Id: extension3ID, Name: "extension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "21779").Return(models.TopGame{Id: "21779", Name: "gameName1", Percentage: 9.531959895581132, Img: "gameImg1"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "138585").Return(models.TopGame{Id: "138585", Name: "gameName2", Percentage: 5.079686475527248, Img: "gameImg2"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "488552").Return(models.TopGame{Id: "488552", Name: "gameName3", Percentage: 4.368633406825281, Img: "gameImg3"}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "32399").Return(models.TopGame{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4885529").Return(models.TopGame{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "217799").Return(models.TopGame{Id: "217799", Name: "gameName6", Percentage: 9.67853060959945, Img: "gameImg6"}, nil)
	suite.discoveryClient.Mock.On("GetGameNameByID", suite.ctx, mainGameID).Return(mainGameName, nil)

	suite.piperDBClient.On("GetGameOnePagerByQuarterAndYear", suite.ctx, mainGameID, suite.startDate).Return(dataBaseReturn, nil)

	suite.cacher.On("GetStringProperties", suite.ctx, suite.dateCacheKey).Return("2018-01-01", true)
	suite.cacher.On("GetStringProperties", suite.ctx, suite.onePagerDataCacheKey).Return("", false)
	suite.cacher.On("CacheStringProperties", suite.ctx, suite.onePagerDataCacheKey, cachedDataStr).Return(nil)

	realResp, err := suite.onePagerImpl.GetGameOnePager(suite.ctx, mainGameID, suite.yearStr, suite.quarterStr)
	suite.NoError(err)

	if diff := utils.Equal(realResp, expectedFullDataResp); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
	suite.discoveryClient.AssertExpectations(suite.T())
}

func (suite *GetInsightsTest) TestDBHasFullDataWithCache() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: true,
		SocialStats:      true,
		ExtensionStats:   true,
		DropStats:        true,
		AudienceStats:    true,
	}

	expectedFullDataResp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, true)
	suite.NoError(err)

	cachedDataStr, err := makeCacheDataString(expectedFullDataResp, presence, true)
	suite.NoError(err)

	suite.cacher.On("GetStringProperties", suite.ctx, suite.dateCacheKey).Return("2018-01-01", true)
	suite.cacher.On("GetStringProperties", suite.ctx, suite.onePagerDataCacheKey).Return(cachedDataStr, true)

	realResp, err := suite.onePagerImpl.GetGameOnePager(suite.ctx, mainGameID, suite.yearStr, suite.quarterStr)
	suite.NoError(err)

	if diff := utils.Equal(realResp, expectedFullDataResp); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
	suite.discoveryClient.AssertExpectations(suite.T())
}

func (suite *GetInsightsTest) TestDBMissingOverviewStats() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    false,
		BroadcasterStats: true,
		SocialStats:      true,
		ExtensionStats:   true,
		DropStats:        true,
		AudienceStats:    true,
	}

	dataBaseReturn, err := makeTestOnePagerDataBasedOnMissingStatus(presence)
	suite.NoError(err)

	expectedFullDataResp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, true)
	suite.NoError(err)

	cachedDataStr, err := makeCacheDataString(expectedFullDataResp, presence, false)
	suite.NoError(err)

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension1ID).Return(models.ExtensionStats{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension2ID).Return(models.ExtensionStats{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension3ID).Return(models.ExtensionStats{Id: extension3ID, Name: "extension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "21779").Return(models.TopGame{Id: "21779", Name: "gameName1", Percentage: 9.531959895581132, Img: "gameImg1"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "138585").Return(models.TopGame{Id: "138585", Name: "gameName2", Percentage: 5.079686475527248, Img: "gameImg2"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "488552").Return(models.TopGame{Id: "488552", Name: "gameName3", Percentage: 4.368633406825281, Img: "gameImg3"}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "32399").Return(models.TopGame{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4885529").Return(models.TopGame{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "217799").Return(models.TopGame{Id: "217799", Name: "gameName6", Percentage: 9.67853060959945, Img: "gameImg6"}, nil)
	suite.discoveryClient.Mock.On("GetGameNameByID", suite.ctx, mainGameID).Return(mainGameName, nil)

	suite.piperDBClient.On("GetGameOnePagerByQuarterAndYear", suite.ctx, mainGameID, suite.startDate).Return(dataBaseReturn, nil)
	suite.piperDBClient.On("GetGameDefaultOnePagerByQuarterAndYear", suite.ctx, suite.startDate).Return(onePagerDefaultData, nil)

	suite.cacher.On("GetStringProperties", suite.ctx, suite.dateCacheKey).Return("2018-01-01", true)
	suite.cacher.On("GetStringProperties", suite.ctx, suite.onePagerDataCacheKey).Return("", false)
	suite.cacher.On("CacheStringProperties", suite.ctx, suite.onePagerDataCacheKey, cachedDataStr).Return(nil)

	realResp, err := suite.onePagerImpl.GetGameOnePager(suite.ctx, mainGameID, suite.yearStr, suite.quarterStr)
	suite.NoError(err)

	if diff := utils.Equal(realResp, expectedFullDataResp); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
	suite.discoveryClient.AssertExpectations(suite.T())
}

func (suite *GetInsightsTest) TestDBMissingBroadcasterStats() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: false,
		SocialStats:      true,
		ExtensionStats:   true,
		DropStats:        true,
		AudienceStats:    true,
	}

	dataBaseReturn, err := makeTestOnePagerDataBasedOnMissingStatus(presence)
	suite.NoError(err)

	expectedFullDataResp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, true)
	suite.NoError(err)

	cachedDataStr, err := makeCacheDataString(expectedFullDataResp, presence, false)
	suite.NoError(err)

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension1ID).Return(models.ExtensionStats{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension2ID).Return(models.ExtensionStats{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension3ID).Return(models.ExtensionStats{Id: extension3ID, Name: "extension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "21779").Return(models.TopGame{Id: "21779", Name: "gameName1", Percentage: 9.531959895581132, Img: "gameImg1"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "138585").Return(models.TopGame{Id: "138585", Name: "gameName2", Percentage: 5.079686475527248, Img: "gameImg2"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "488552").Return(models.TopGame{Id: "488552", Name: "gameName3", Percentage: 4.368633406825281, Img: "gameImg3"}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "32399").Return(models.TopGame{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4885529").Return(models.TopGame{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "217799").Return(models.TopGame{Id: "217799", Name: "gameName6", Percentage: 9.67853060959945, Img: "gameImg6"}, nil)
	suite.discoveryClient.Mock.On("GetGameNameByID", suite.ctx, mainGameID).Return(mainGameName, nil)

	suite.piperDBClient.On("GetGameOnePagerByQuarterAndYear", suite.ctx, mainGameID, suite.startDate).Return(dataBaseReturn, nil)
	suite.piperDBClient.On("GetGameDefaultOnePagerByQuarterAndYear", suite.ctx, suite.startDate).Return(onePagerDefaultData, nil)

	suite.cacher.On("GetStringProperties", suite.ctx, suite.dateCacheKey).Return("2018-01-01", true)
	suite.cacher.On("GetStringProperties", suite.ctx, suite.onePagerDataCacheKey).Return("", false)
	suite.cacher.On("CacheStringProperties", suite.ctx, suite.onePagerDataCacheKey, cachedDataStr).Return(nil)

	realResp, err := suite.onePagerImpl.GetGameOnePager(suite.ctx, mainGameID, suite.yearStr, suite.quarterStr)
	suite.NoError(err)

	if diff := utils.Equal(realResp, expectedFullDataResp); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
	suite.discoveryClient.AssertExpectations(suite.T())
}

func (suite *GetInsightsTest) TestDBMissingSocialStats() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: true,
		SocialStats:      false,
		ExtensionStats:   true,
		DropStats:        true,
		AudienceStats:    true,
	}

	dataBaseReturn, err := makeTestOnePagerDataBasedOnMissingStatus(presence)
	suite.NoError(err)

	expectedFullDataResp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, true)
	suite.NoError(err)

	cachedDataStr, err := makeCacheDataString(expectedFullDataResp, presence, false)
	suite.NoError(err)

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension1ID).Return(models.ExtensionStats{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension2ID).Return(models.ExtensionStats{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension3ID).Return(models.ExtensionStats{Id: extension3ID, Name: "extension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "21779").Return(models.TopGame{Id: "21779", Name: "gameName1", Percentage: 9.531959895581132, Img: "gameImg1"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "138585").Return(models.TopGame{Id: "138585", Name: "gameName2", Percentage: 5.079686475527248, Img: "gameImg2"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "488552").Return(models.TopGame{Id: "488552", Name: "gameName3", Percentage: 4.368633406825281, Img: "gameImg3"}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "32399").Return(models.TopGame{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4885529").Return(models.TopGame{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "217799").Return(models.TopGame{Id: "217799", Name: "gameName6", Percentage: 9.67853060959945, Img: "gameImg6"}, nil)
	suite.discoveryClient.Mock.On("GetGameNameByID", suite.ctx, mainGameID).Return(mainGameName, nil)

	suite.piperDBClient.On("GetGameOnePagerByQuarterAndYear", suite.ctx, mainGameID, suite.startDate).Return(dataBaseReturn, nil)
	suite.piperDBClient.On("GetGameDefaultOnePagerByQuarterAndYear", suite.ctx, suite.startDate).Return(onePagerDefaultData, nil)

	suite.cacher.On("GetStringProperties", suite.ctx, suite.dateCacheKey).Return("2018-01-01", true)
	suite.cacher.On("GetStringProperties", suite.ctx, suite.onePagerDataCacheKey).Return("", false)
	suite.cacher.On("CacheStringProperties", suite.ctx, suite.onePagerDataCacheKey, cachedDataStr).Return(nil)

	realResp, err := suite.onePagerImpl.GetGameOnePager(suite.ctx, mainGameID, suite.yearStr, suite.quarterStr)
	suite.NoError(err)

	if diff := utils.Equal(realResp, expectedFullDataResp); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
	suite.discoveryClient.AssertExpectations(suite.T())
}

func (suite *GetInsightsTest) TestDBMissingExtensionStats() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: true,
		SocialStats:      true,
		ExtensionStats:   false,
		DropStats:        true,
		AudienceStats:    true,
	}

	dataBaseReturn, err := makeTestOnePagerDataBasedOnMissingStatus(presence)
	suite.NoError(err)

	expectedFullDataResp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, true)
	suite.NoError(err)

	cachedDataStr, err := makeCacheDataString(expectedFullDataResp, presence, false)
	suite.NoError(err)

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, "2b9s9g75awk0ju391l0o2w4e7jxo47").Return(models.ExtensionStats{Id: "2b9s9g75awk0ju391l0o2w4e7jxo47", Name: "popularExtension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, "gh70y1spw727ohtgzbhc0hppvq9br2").Return(models.ExtensionStats{Id: "gh70y1spw727ohtgzbhc0hppvq9br2", Name: "popularExtension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, "wi08ebtatdc7oj83wtl9uxwz807l8b").Return(models.ExtensionStats{Id: "wi08ebtatdc7oj83wtl9uxwz807l8b", Name: "popularExtension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "21779").Return(models.TopGame{Id: "21779", Name: "gameName1", Percentage: 9.531959895581132, Img: "gameImg1"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "138585").Return(models.TopGame{Id: "138585", Name: "gameName2", Percentage: 5.079686475527248, Img: "gameImg2"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "488552").Return(models.TopGame{Id: "488552", Name: "gameName3", Percentage: 4.368633406825281, Img: "gameImg3"}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "32399").Return(models.TopGame{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4885529").Return(models.TopGame{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "217799").Return(models.TopGame{Id: "217799", Name: "gameName6", Percentage: 9.67853060959945, Img: "gameImg6"}, nil)

	suite.discoveryClient.Mock.On("GetGameNameByID", suite.ctx, mainGameID).Return(mainGameName, nil)

	suite.piperDBClient.On("GetGameOnePagerByQuarterAndYear", suite.ctx, mainGameID, suite.startDate).Return(dataBaseReturn, nil)
	suite.piperDBClient.On("GetGameDefaultOnePagerByQuarterAndYear", suite.ctx, suite.startDate).Return(onePagerDefaultData, nil)

	suite.cacher.On("GetStringProperties", suite.ctx, suite.dateCacheKey).Return("2018-01-01", true)
	suite.cacher.On("GetStringProperties", suite.ctx, suite.onePagerDataCacheKey).Return("", false)
	suite.cacher.On("CacheStringProperties", suite.ctx, suite.onePagerDataCacheKey, cachedDataStr).Return(nil)

	realResp, err := suite.onePagerImpl.GetGameOnePager(suite.ctx, mainGameID, suite.yearStr, suite.quarterStr)
	suite.NoError(err)

	if diff := utils.Equal(realResp, expectedFullDataResp); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
	suite.discoveryClient.AssertExpectations(suite.T())
}

func (suite *GetInsightsTest) TestDBMissingDropsStats() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: true,
		SocialStats:      true,
		ExtensionStats:   true,
		DropStats:        false,
		AudienceStats:    true,
	}

	dataBaseReturn, err := makeTestOnePagerDataBasedOnMissingStatus(presence)
	suite.NoError(err)

	expectedFullDataResp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, true)
	suite.NoError(err)

	cachedDataStr, err := makeCacheDataString(expectedFullDataResp, presence, false)
	suite.NoError(err)

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension1ID).Return(models.ExtensionStats{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension2ID).Return(models.ExtensionStats{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension3ID).Return(models.ExtensionStats{Id: extension3ID, Name: "extension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "21779").Return(models.TopGame{Id: "21779", Name: "gameName1", Percentage: 9.531959895581132, Img: "gameImg1"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "138585").Return(models.TopGame{Id: "138585", Name: "gameName2", Percentage: 5.079686475527248, Img: "gameImg2"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "488552").Return(models.TopGame{Id: "488552", Name: "gameName3", Percentage: 4.368633406825281, Img: "gameImg3"}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "32399").Return(models.TopGame{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4885529").Return(models.TopGame{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "217799").Return(models.TopGame{Id: "217799", Name: "gameName6", Percentage: 9.67853060959945, Img: "gameImg6"}, nil)
	suite.discoveryClient.Mock.On("GetGameNameByID", suite.ctx, mainGameID).Return(mainGameName, nil)

	suite.piperDBClient.On("GetGameOnePagerByQuarterAndYear", suite.ctx, mainGameID, suite.startDate).Return(dataBaseReturn, nil)
	suite.piperDBClient.On("GetGameDefaultOnePagerByQuarterAndYear", suite.ctx, suite.startDate).Return(onePagerDefaultData, nil)

	suite.cacher.On("GetStringProperties", suite.ctx, suite.dateCacheKey).Return("2018-01-01", true)
	suite.cacher.On("GetStringProperties", suite.ctx, suite.onePagerDataCacheKey).Return("", false)
	suite.cacher.On("CacheStringProperties", suite.ctx, suite.onePagerDataCacheKey, cachedDataStr).Return(nil)

	realResp, err := suite.onePagerImpl.GetGameOnePager(suite.ctx, mainGameID, suite.yearStr, suite.quarterStr)
	suite.NoError(err)

	if diff := utils.Equal(realResp, expectedFullDataResp); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
	suite.discoveryClient.AssertExpectations(suite.T())
}

func (suite *GetInsightsTest) TestDBMissingAudienceStats() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: true,
		SocialStats:      true,
		ExtensionStats:   true,
		DropStats:        true,
		AudienceStats:    false,
	}

	dataBaseReturn, err := makeTestOnePagerDataBasedOnMissingStatus(presence)
	suite.NoError(err)

	expectedFullDataResp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, true)
	suite.NoError(err)

	cachedDataStr, err := makeCacheDataString(expectedFullDataResp, presence, false)
	suite.NoError(err)

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension1ID).Return(models.ExtensionStats{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension2ID).Return(models.ExtensionStats{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension3ID).Return(models.ExtensionStats{Id: extension3ID, Name: "extension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "1458").Return(models.TopGame{Id: "1458", Name: "popularGameName1", Percentage: 45, Img: "popularGameImg1"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4609").Return(models.TopGame{Id: "4609", Name: "popularGameName2", Percentage: 28.1, Img: "popularGameImg2"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "3734").Return(models.TopGame{Id: "3734", Name: "popularGameName3", Percentage: 10.28, Img: "popularGameImg3"}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "46091").Return(models.TopGame{Id: "46091", Name: "popularGameName4", Percentage: 38.12, Img: "popularGameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "582").Return(models.TopGame{Id: "582", Name: "popularGameName5", Percentage: 32.19, Img: "popularGameImg5"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "37341").Return(models.TopGame{Id: "37341", Name: "popularGameName6", Percentage: 12.28, Img: "popularGameImg6"}, nil)

	suite.discoveryClient.Mock.On("GetGameNameByID", suite.ctx, mainGameID).Return(mainGameName, nil)

	suite.piperDBClient.On("GetGameOnePagerByQuarterAndYear", suite.ctx, mainGameID, suite.startDate).Return(dataBaseReturn, nil)
	suite.piperDBClient.On("GetGameDefaultOnePagerByQuarterAndYear", suite.ctx, suite.startDate).Return(onePagerDefaultData, nil)

	suite.cacher.On("GetStringProperties", suite.ctx, suite.dateCacheKey).Return("2018-01-01", true)
	suite.cacher.On("GetStringProperties", suite.ctx, suite.onePagerDataCacheKey).Return("", false)
	suite.cacher.On("CacheStringProperties", suite.ctx, suite.onePagerDataCacheKey, cachedDataStr).Return(nil)

	realResp, err := suite.onePagerImpl.GetGameOnePager(suite.ctx, mainGameID, suite.yearStr, suite.quarterStr)
	suite.NoError(err)

	if diff := utils.Equal(realResp, expectedFullDataResp); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
	suite.discoveryClient.AssertExpectations(suite.T())
}

func (suite *GetInsightsTest) TestDBMissingAllData() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    false,
		BroadcasterStats: false,
		SocialStats:      false,
		ExtensionStats:   false,
		DropStats:        false,
		AudienceStats:    false,
	}

	dataBaseReturn, err := makeTestOnePagerDataBasedOnMissingStatus(presence)
	suite.NoError(err)

	expectedFullDataResp, err := makeTestOnePagerResponseWithMissingStatus(presence, true, true)
	suite.NoError(err)

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, "2b9s9g75awk0ju391l0o2w4e7jxo47").Return(models.ExtensionStats{Id: "2b9s9g75awk0ju391l0o2w4e7jxo47", Name: "popularExtension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, "gh70y1spw727ohtgzbhc0hppvq9br2").Return(models.ExtensionStats{Id: "gh70y1spw727ohtgzbhc0hppvq9br2", Name: "popularExtension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, "wi08ebtatdc7oj83wtl9uxwz807l8b").Return(models.ExtensionStats{Id: "wi08ebtatdc7oj83wtl9uxwz807l8b", Name: "popularExtension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "1458").Return(models.TopGame{Id: "1458", Name: "popularGameName1", Percentage: 45, Img: "popularGameImg1"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4609").Return(models.TopGame{Id: "4609", Name: "popularGameName2", Percentage: 28.1, Img: "popularGameImg2"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "3734").Return(models.TopGame{Id: "3734", Name: "popularGameName3", Percentage: 10.28, Img: "popularGameImg3"}, nil)

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "46091").Return(models.TopGame{Id: "46091", Name: "popularGameName4", Percentage: 38.12, Img: "popularGameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "582").Return(models.TopGame{Id: "582", Name: "popularGameName5", Percentage: 32.19, Img: "popularGameImg5"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "37341").Return(models.TopGame{Id: "37341", Name: "popularGameName6", Percentage: 12.28, Img: "popularGameImg6"}, nil)

	suite.discoveryClient.Mock.On("GetGameNameByID", suite.ctx, mainGameID).Return(mainGameName, nil)

	suite.piperDBClient.On("GetGameOnePagerByQuarterAndYear", suite.ctx, mainGameID, suite.startDate).Return(dataBaseReturn, nil)
	suite.piperDBClient.On("GetGameDefaultOnePagerByQuarterAndYear", suite.ctx, suite.startDate).Return(onePagerDefaultData, nil)

	suite.cacher.On("GetStringProperties", suite.ctx, suite.dateCacheKey).Return("2018-01-01", true)
	suite.cacher.On("GetStringProperties", suite.ctx, suite.onePagerDataCacheKey).Return("", false)

	realResp, err := suite.onePagerImpl.GetGameOnePager(suite.ctx, mainGameID, suite.yearStr, suite.quarterStr)
	suite.NoError(err)

	if diff := utils.Equal(realResp, expectedFullDataResp); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
	suite.discoveryClient.AssertExpectations(suite.T())
}

func TestGetInsightsSuite(t *testing.T) {
	suite.Run(t, new(GetInsightsTest))
}
