package one_pager

import (
	"testing"

	"code.justin.tv/insights/piper-service/internal/utils"
	"code.justin.tv/insights/piper-service/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/stretchr/testify/suite"
)

type ParseExtensionTest struct {
	onePagerSuite
}

func (suite *ParseExtensionTest) TestParseExtensionSuccess() {

	presence := &models.OnePagerSectionPresentStatus{ExtensionStats: true}

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension1ID).Return(models.ExtensionStats{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension2ID).Return(models.ExtensionStats{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension3ID).Return(models.ExtensionStats{Id: extension3ID, Name: "extension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")}, nil)

	topExtensions := suite.onePagerImpl.parseExtensions(suite.ctx, topExtensions, presence)

	if diff := utils.Equal(topExtensions, *onePagerResponseWithFullData.ExtensionStats); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
}

func (suite *ParseExtensionTest) TestParseExtensionWithOnlyTwoExtensions() {

	presence := &models.OnePagerSectionPresentStatus{ExtensionStats: true}

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension1ID).Return(models.ExtensionStats{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension2ID).Return(models.ExtensionStats{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)

	topExtensions := suite.onePagerImpl.parseExtensions(suite.ctx, "wi08ebtatdc7oj83wtl9uxwz807l8b||7hoqd16sufw9f9h3swlkdt6lqdn3ud", presence)
	if diff := utils.Equal(topExtensions, []models.ExtensionStats{
		{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")},
		{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")},
	}); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
}

func (suite *ParseExtensionTest) TestParseWithOnlyOneExtensions() {

	presence := &models.OnePagerSectionPresentStatus{ExtensionStats: true}

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension1ID).Return(models.ExtensionStats{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)

	topExtensions := suite.onePagerImpl.parseExtensions(suite.ctx, "wi08ebtatdc7oj83wtl9uxwz807l8b", presence)
	if diff := utils.Equal(topExtensions, []models.ExtensionStats{
		{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")},
	}); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
}

func (suite *ParseExtensionTest) TestParseWithOnlyOneEMSServiceFailure() {
	presence := &models.OnePagerSectionPresentStatus{ExtensionStats: true}

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension1ID).Return(models.ExtensionStats{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension2ID).Return(models.ExtensionStats{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension3ID).Return(models.ExtensionStats{}, models.ErrResourceNotFound)

	topExtensions := suite.onePagerImpl.parseExtensions(suite.ctx, topExtensions, presence)
	if diff := utils.Equal(topExtensions, []models.ExtensionStats{
		{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")},
		{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")},
	}); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
}

func (suite *ParseExtensionTest) TestParseWithOnlyPartialBadString() {
	presence := &models.OnePagerSectionPresentStatus{ExtensionStats: true}

	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension1ID).Return(models.ExtensionStats{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")}, nil)
	suite.emsClient.Mock.On("GetReleasedExtensionByID", suite.ctx, extension2ID).Return(models.ExtensionStats{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")}, nil)

	topExtensions := suite.onePagerImpl.parseExtensions(suite.ctx, "wi08ebtatdc7oj83wtl9uxwz807l8b||7hoqd16sufw9f9h3swlkdt6lqdn3ud||", presence)
	if diff := utils.Equal(topExtensions, []models.ExtensionStats{
		{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")},
		{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")},
	}); diff != nil {
		suite.T().Error(diff)
	}

	suite.emsClient.AssertExpectations(suite.T())
}

func TestParseExtensionSuite(t *testing.T) {
	suite.Run(t, new(ParseExtensionTest))
}
