package one_pager

import (
	"testing"

	"code.justin.tv/insights/piper-service/internal/utils"
	"code.justin.tv/insights/piper-service/models"
	"github.com/stretchr/testify/suite"
)

type ParseGameTest struct {
	onePagerSuite
}

func (suite *ParseGameTest) TestParseGameSuccess() {

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "32399").Return(models.TopGame{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4885529").Return(models.TopGame{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "217799").Return(models.TopGame{Id: "217799", Name: "gameName6", Percentage: 9.67853060959945, Img: "gameImg6"}, nil)

	topGames := suite.onePagerImpl.parseGame(suite.ctx, topThreeGamesByStreamers)
	if diff := utils.Equal(topGames, onePagerResponseWithFullData.AudienceStats.TopGamesToStream); diff != nil {
		suite.T().Error(diff)
	}

	suite.discoveryClient.AssertExpectations(suite.T())
}

func (suite *ParseGameTest) TestParseGameWithOnlyTwoGames() {

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "32399").Return(models.TopGame{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4885529").Return(models.TopGame{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"}, nil)

	topGames := suite.onePagerImpl.parseGame(suite.ctx, "32399,14.848803747515351||4885529,9.942663558381568")
	if diff := utils.Equal(topGames, []models.TopGame{
		{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"},
		{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"},
	}); diff != nil {
		suite.T().Error(diff)
	}

	suite.discoveryClient.AssertExpectations(suite.T())
}

func (suite *ParseGameTest) TestParseGameWithOnlyOneGames() {

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "32399").Return(models.TopGame{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"}, nil)

	topGames := suite.onePagerImpl.parseGame(suite.ctx, "32399,14.848803747515351")
	if diff := utils.Equal(topGames, []models.TopGame{
		{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"},
	}); diff != nil {
		suite.T().Error(diff)
	}

	suite.discoveryClient.AssertExpectations(suite.T())
}

func (suite *ParseGameTest) TestParseGameWithBrokenString() {
	topGames := suite.onePagerImpl.parseGame(suite.ctx, "32399")
	if diff := utils.Equal(topGames, []models.TopGame{}); diff != nil {
		suite.T().Error(diff)
	}
}

func (suite *ParseGameTest) TestParseGameWithOnlyOneDiscoveryServiceFailure() {

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "32399").Return(models.TopGame{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4885529").Return(models.TopGame{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "217799").Return(models.TopGame{}, models.ErrGameNotFound)

	topGames := suite.onePagerImpl.parseGame(suite.ctx, topThreeGamesByStreamers)
	if diff := utils.Equal(topGames, []models.TopGame{
		{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"},
		{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"},
	}); diff != nil {
		suite.T().Error(diff)
	}

	suite.discoveryClient.AssertExpectations(suite.T())
}

func (suite *ParseGameTest) TestParseGameWithOnlyPartialBadString() {

	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "32399").Return(models.TopGame{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"}, nil)
	suite.discoveryClient.Mock.On("GetGameByID", suite.ctx, "4885529").Return(models.TopGame{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"}, nil)

	topGames := suite.onePagerImpl.parseGame(suite.ctx, "32399,14.848803747515351||2390,xowie||4885529,9.942663558381568")
	if diff := utils.Equal(topGames, []models.TopGame{
		{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"},
		{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"},
	}); diff != nil {
		suite.T().Error(diff)
	}

	suite.discoveryClient.AssertExpectations(suite.T())
}

func TestParseGameSuite(t *testing.T) {
	suite.Run(t, new(ParseGameTest))
}
