package one_pager

import (
	"encoding/json"

	"code.justin.tv/insights/piper-service/internal/utils"
	"code.justin.tv/insights/piper-service/models"
	"github.com/aws/aws-sdk-go/aws"
)

const (
	mainGameName = "Fortnite"
	mainGameID   = "33214"

	extension1ID  = "wi08ebtatdc7oj83wtl9uxwz807l8b"
	extension2ID  = "7hoqd16sufw9f9h3swlkdt6lqdn3ud"
	extension3ID  = "9zqhb3hba48m0c382uno7k2t74isfi"
	topExtensions = "wi08ebtatdc7oj83wtl9uxwz807l8b||7hoqd16sufw9f9h3swlkdt6lqdn3ud||9zqhb3hba48m0c382uno7k2t74isfi"

	topThreeGamesByStreamers = "32399,14.848803747515351||4885529,9.942663558381568||217799,9.67853060959945"
	topThreeGamesByViewers   = "21779,9.531959895581132||138585,5.079686475527248||488552,4.368633406825281"
	risingCountryByStreamers = "US,United States,9.020||JP,Japan,5.66666666667||AF,Afghanistan,1.69230769231"
	risingCountryByViewers   = "US,United States,3.50||DZ,Algeria,3.12564302379||AX,Aland Islands,2.94245969016"
)

var onePagerWithAllData = models.OnePagerData{
	HoursBroadcastCurrent:       145.1833343506,
	HoursBroadcastQoQ:           8.1833333969,
	HoursWatchedCurrent:         72.9666671753,
	HoursWatchedQoQ:             -15.6464138031,
	UniqueBroadcastersCurrent:   2541,
	UniqueBroadcastersQoQ:       307.8999938965,
	UniqueViewersCurrent:        3921,
	UniqueViewersQoQ:            -50.7332572937,
	BroadcastingSessionsCurrent: 15,
	BroadcastingSessionsQoQ:     19,
	ViewingSessionsCurrent:      2019,
	ViewingSessionsQoQ:          725.3978271484,

	TopChannelLogin:                    "youngbeingyoung",
	TopChannelDisplayName:              "YoungBeingYoung",
	TopChannelHoursBroadcast:           8.0833330154,
	TopChannelHoursWatched:             0.1333333403,
	TopChannelAvgCCU:                   10,
	RisingChannelLogin:                 "riotgames",
	RisingChannelDisplayname:           "Riot Games",
	RisingChannelHoursBroadcast:        8.0833330154,
	RisingChannelHoursWatched:          0.1333333403,
	RisingChannelAvgCCU:                27,
	MostDedicatedChannelLogin:          "mia1912",
	MostDedicatedChannelDisplayname:    "용연향",
	MostDedicatedChannelSessions:       4,
	MostDedicatedChannelGamePercentage: 80.941231,

	TotalChatMessages:      502932,
	UniqueChatters:         21750,
	ChatterPercent:         35.3269844055,
	UniquePartnerAffiliate: 1600,

	TotalActiveExtensions: 4236,
	MostPopularExtensions: topExtensions,

	DropsReceivedCurrent:              982109,
	DropsReceivedQoQ:                  -54.9283211,
	UniqueViewersReceivedADropCurrent: 192811,
	UniqueViewersReceivedADropQoQ:     29.19281837,
	ActiveCampaignsCurrent:            6,
	ActiveCampaignsQoQ:                -1.6247804165,

	RisingCountriesByStreamers: risingCountryByStreamers,
	RisingCountriesByViewers:   risingCountryByViewers,
	TopThreeGamesByStreamers:   topThreeGamesByStreamers,
	TopThreeGamesByViewers:     topThreeGamesByViewers,
}

var onePagerResponseWithFullData = models.GameOnePagerDataResponse{
	GameInfo: models.GameInfo{},

	NumOfExtensions: 4236,

	OverviewStats: &models.OverviewStats{
		HoursWatched:                onePagerWithAllData.HoursWatchedCurrent,
		HoursWatchedChanges:         onePagerWithAllData.HoursWatchedQoQ,
		UniqueViewers:               onePagerWithAllData.UniqueViewersCurrent,
		UniqueViewersChanges:        onePagerWithAllData.UniqueViewersQoQ,
		ViewingSessions:             onePagerWithAllData.ViewingSessionsCurrent,
		ViewingSessionsChanges:      onePagerWithAllData.ViewingSessionsQoQ,
		HoursBroadcast:              onePagerWithAllData.HoursBroadcastCurrent,
		HoursBroadcastChanges:       onePagerWithAllData.HoursBroadcastQoQ,
		UniqueBroadcasters:          onePagerWithAllData.UniqueBroadcastersCurrent,
		UniqueBroadcastersChanges:   onePagerWithAllData.UniqueBroadcastersQoQ,
		BroadcastingSessions:        onePagerWithAllData.BroadcastingSessionsCurrent,
		BroadcastingSessionsChanges: onePagerWithAllData.BroadcastingSessionsQoQ,
	},

	BroadcasterStats: &models.BroadcasterStats{
		TopBroadcaster: &models.TopBroadcaster{
			Name:              onePagerWithAllData.TopChannelLogin,
			DisplayName:       onePagerWithAllData.TopChannelDisplayName,
			HoursBroadcast:    onePagerWithAllData.TopChannelHoursBroadcast,
			HoursWatched:      onePagerWithAllData.TopChannelHoursWatched,
			ConcurrentViewers: onePagerWithAllData.TopChannelAvgCCU,
		},
		DedicatedBroadcaster: &models.DedicatedBroadcaster{
			Name:                       onePagerWithAllData.MostDedicatedChannelLogin,
			DisplayName:                onePagerWithAllData.MostDedicatedChannelDisplayname,
			StreamedSessions:           onePagerWithAllData.MostDedicatedChannelSessions,
			AvgStreamSessionsPerWeek:   0.33,
			PercentageOfHoursBroadcast: onePagerWithAllData.MostDedicatedChannelGamePercentage,
		},
		RisingStar: &models.RisingStar{
			Name:                 onePagerWithAllData.RisingChannelLogin,
			DisplayName:          onePagerWithAllData.RisingChannelDisplayname,
			HoursStreamed:        onePagerWithAllData.RisingChannelHoursBroadcast,
			HoursWatched:         onePagerWithAllData.RisingChannelHoursWatched,
			AvgConcurrentViewers: onePagerWithAllData.RisingChannelAvgCCU,
		},
	},

	SocialStats: &models.SocialStats{
		NumOfChatMessages:     onePagerWithAllData.TotalChatMessages,
		PercentOfEngagement:   onePagerWithAllData.ChatterPercent,
		NumOfPartnersStreamed: onePagerWithAllData.UniquePartnerAffiliate,
	},

	ExtensionStats: &[]models.ExtensionStats{
		{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")},
		{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")},
		{Id: extension3ID, Name: "extension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")},
	},

	DropsStats: &models.DropStats{
		NumOfDropsReceived:               onePagerWithAllData.DropsReceivedCurrent,
		NumOfDropsReceivedChange:         onePagerWithAllData.DropsReceivedQoQ,
		NumOfUniqueReceivingUsers:        onePagerWithAllData.UniqueViewersReceivedADropCurrent,
		NumOfUniqueReceivingUsersChanges: onePagerWithAllData.UniqueViewersReceivedADropQoQ,
		NumOfCampaigns:                   onePagerWithAllData.ActiveCampaignsCurrent,
		NumOfCampaignsChanges:            onePagerWithAllData.ActiveCampaignsQoQ,
	},

	AudienceStats: &models.AudienceStats{
		TopGamesToWatch: []models.TopGame{
			{Id: "21779", Name: "gameName1", Percentage: 9.531959895581132, Img: "gameImg1"},
			{Id: "138585", Name: "gameName2", Percentage: 5.079686475527248, Img: "gameImg2"},
			{Id: "488552", Name: "gameName3", Percentage: 4.368633406825281, Img: "gameImg3"},
		},
		TopGamesToStream: []models.TopGame{
			{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"},
			{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"},
			{Id: "217799", Name: "gameName6", Percentage: 9.67853060959945, Img: "gameImg6"},
		},
		TopBroadcasterCountries: []models.TopCountry{
			{Name: "United States", Percentage: 9.020, Code: "US"},
			{Name: "Japan", Percentage: 5.66666666667, Code: "JP"},
			{Name: "Afghanistan", Percentage: 1.69230769231, Code: "AF"},
		},
		TopViewerCountries: []models.TopCountry{
			{Name: "United States", Percentage: 3.50, Code: "US"},
			{Name: "Algeria", Percentage: 3.12564302379, Code: "DZ"},
			{Name: "Aland Islands", Percentage: 2.94245969016, Code: "AX"},
		},
	},

	GeneralStats:        nil,
	PopularBroadcasters: nil,
	PopularExtensions:   nil,
	PopularGames:        nil,
}

var onePagerDefaultDataResp = models.GameOnePagerDataResponse{
	NumOfExtensions:     onePagerDefaultData.TotalActiveExtensions,
	GeneralStats:        &generalStats,
	PopularBroadcasters: &popularBroadcasters,
	PopularExtensions:   &popularExtensions,
	PopularGames:        &popularGames,
}

var generalStats = models.TwitchOverviewStats{
	MinutesWatched:            355000000000,
	UniqueMonthlyBroadcasters: 1500000,
	UniqueDailyVisitors:       15000000,
	PartneredStreamers:        27000000,
	AffiliateStreamers:        150000,
	TotalClips:                124000000,
}

var popularBroadcasters = []models.PopularBroadcaster{
	{Name: "anbornsky", DisplayName: "AnbornSky"},
	{Name: "xocliw", DisplayName: "Xocliw"},
	{Name: "theplastereddragon", DisplayName: "theplastereddragon"},
}

var popularExtensions = []models.ExtensionStats{
	{Id: "2b9s9g75awk0ju391l0o2w4e7jxo47", Name: "popularExtension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")},
	{Id: "gh70y1spw727ohtgzbhc0hppvq9br2", Name: "popularExtension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")},
	{Id: "wi08ebtatdc7oj83wtl9uxwz807l8b", Name: "popularExtension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")},
}

var popularGames = models.PopularGames{
	PopularGamesToStream: []models.TwitchPopularGame{
		{Id: "1458", Name: "popularGameName1", Img: "popularGameImg1"},
		{Id: "4609", Name: "popularGameName2", Img: "popularGameImg2"},
		{Id: "3734", Name: "popularGameName3", Img: "popularGameImg3"},
	},
	PopularGamesToWatch: []models.TwitchPopularGame{
		{Id: "46091", Name: "popularGameName4", Img: "popularGameImg4"},
		{Id: "582", Name: "popularGameName5", Img: "popularGameImg5"},
		{Id: "37341", Name: "popularGameName6", Img: "popularGameImg6"},
	},
}

var onePagerDefaultData = models.OnePagerDefaultData{
	MinutesWatched:            355000000000,
	UniqueMonthlyBroadcasters: 1500000,
	UniqueDailyVisitors:       15000000,
	PartneredStreamers:        27000000,
	AffiliateStreamers:        150000,
	TotalClips:                124000000,
	TotalActiveExtensions:     4236,
	TopThreeBroadcaster:       "anbornsky,AnbornSky||xocliw,Xocliw||theplastereddragon,theplastereddragon",
	TopThreeExtensions:        "2b9s9g75awk0ju391l0o2w4e7jxo47||gh70y1spw727ohtgzbhc0hppvq9br2||wi08ebtatdc7oj83wtl9uxwz807l8b",
	TopThreeGamesByStreamers:  "1458,45||4609,28.1||3734,10.28",
	TopThreeGamesByViewers:    "46091,38.12||582,32.19||37341,12.28",
}

type overviewStats struct {
	HoursBroadcastCurrent       float64 `json:"hours_broadcast_current"`
	HoursBroadcastQoQ           float64 `json:"hours_broadcast_qoq"`
	HoursWatchedCurrent         float64 `json:"hours_watched_current"`
	HoursWatchedQoQ             float64 `json:"hours_watched_qoq"`
	UniqueBroadcastersCurrent   uint64  `json:"unique_broadcasters_current"`
	UniqueBroadcastersQoQ       float64 `json:"unique_broadcasters_qoq"`
	UniqueViewersCurrent        uint64  `json:"unique_viewers_current"`
	UniqueViewersQoQ            float64 `json:"unique_viewers_qoq"`
	BroadcastingSessionsCurrent uint64  `json:"broadcasting_sessions_current"`
	BroadcastingSessionsQoQ     float64 `json:"broadcasting_sessions_qoq"`
	ViewingSessionsCurrent      uint64  `json:"viewing_sessions_current"`
	ViewingSessionsQoQ          float64 `json:"viewing_sessions_qoq"`
}

type audienceStats struct {
	RisingCountriesByStreamers string `json:"rising_countries_by_streamers"`
	RisingCountriesByViewers   string `json:"rising_countries_by_viewers"`
	TopThreeGamesByStreamers   string `json:"top_three_games_by_streamers"`
	TopThreeGamesByViewers     string `json:"top_three_games_by_viewers"`
}

type dropsStats struct {
	DropsReceivedCurrent              uint64  `json:"drops_received_current"`
	DropsReceivedQoQ                  float64 `json:"drops_received_qoq"`
	UniqueViewersReceivedADropCurrent uint64  `json:"unique_viewers_received_a_drop_current"`
	UniqueViewersReceivedADropQoQ     float64 `json:"unique_viewers_received_a_drop_qoq"`
	ActiveCampaignsCurrent            uint64  `json:"active_campaigns_current"`
	ActiveCampaignsQoQ                float64 `json:"active_campaigns_qoq"`
}

type broadcasterStats struct {
	TopChannelLogin                    string  `json:"top_channel_login"`
	TopChannelDisplayName              string  `json:"top_channel_displayname"`
	TopChannelHoursBroadcast           float64 `json:"top_channel_hours_broadcast"`
	TopChannelHoursWatched             float64 `json:"top_channel_hours_watched"`
	TopChannelAvgCCU                   uint64  `json:"top_channel_avg_ccu"`
	RisingChannelLogin                 string  `json:"rising_channel_login"`
	RisingChannelDisplayname           string  `json:"rising_channel_displayname"`
	RisingChannelHoursBroadcast        float64 `json:"rising_channel_hours_broadcast"`
	RisingChannelHoursWatched          float64 `json:"rising_channel_hours_watched"`
	RisingChannelAvgCCU                uint64  `json:"rising_channel_avg_ccu"`
	MostDedicatedChannelLogin          string  `json:"most_dedicated_channel_login"`
	MostDedicatedChannelDisplayname    string  `json:"most_dedicated_channel_displayname"`
	MostDedicatedChannelSessions       uint64  `json:"most_dedicated_channel_sessions"`
	MostDedicatedChannelGamePercentage float64 `json:"most_dedicated_game_percentage"`
}

func makeTestOnePagerDataBasedOnMissingStatus(status models.OnePagerSectionPresentStatus) (models.OnePagerData, error) {
	copiedData := models.OnePagerData{}
	err := utils.Copy(&copiedData, &onePagerWithAllData)
	if err != nil {
		return copiedData, err
	}

	copiedData.TotalActiveExtensions = onePagerDefaultData.TotalActiveExtensions

	if !status.OverviewStats {
		overviewStats := overviewStats{
			HoursBroadcastCurrent:       missingDataNum,
			HoursBroadcastQoQ:           missingDataNum,
			HoursWatchedCurrent:         missingDataNum,
			HoursWatchedQoQ:             missingDataNum,
			UniqueBroadcastersCurrent:   missingDataNum,
			UniqueBroadcastersQoQ:       missingDataNum,
			UniqueViewersCurrent:        missingDataNum,
			UniqueViewersQoQ:            missingDataNum,
			BroadcastingSessionsCurrent: missingDataNum,
			BroadcastingSessionsQoQ:     missingDataNum,
			ViewingSessionsCurrent:      missingDataNum,
			ViewingSessionsQoQ:          missingDataNum,
		}
		err := utils.Copy(&copiedData, &overviewStats)
		if err != nil {
			return copiedData, err
		}
	}

	if !status.BroadcasterStats {
		broadcasterStats := broadcasterStats{
			TopChannelLogin:                    missingDataStr,
			TopChannelDisplayName:              missingDataStr,
			TopChannelHoursBroadcast:           missingDataNum,
			TopChannelHoursWatched:             missingDataNum,
			TopChannelAvgCCU:                   missingDataNum,
			RisingChannelLogin:                 missingDataStr,
			RisingChannelDisplayname:           missingDataStr,
			RisingChannelHoursBroadcast:        missingDataNum,
			RisingChannelHoursWatched:          missingDataNum,
			RisingChannelAvgCCU:                missingDataNum,
			MostDedicatedChannelLogin:          missingDataStr,
			MostDedicatedChannelDisplayname:    missingDataStr,
			MostDedicatedChannelSessions:       missingDataNum,
			MostDedicatedChannelGamePercentage: missingDataNum,
		}
		err := utils.Copy(&copiedData, &broadcasterStats)
		if err != nil {
			return copiedData, err
		}
	}

	if !status.SocialStats {
		copiedData.TotalChatMessages = missingDataNum
		copiedData.UniqueChatters = missingDataNum
		copiedData.ChatterPercent = missingDataNum
		copiedData.UniquePartnerAffiliate = missingDataNum
	}

	if !status.ExtensionStats {
		copiedData.MostPopularExtensions = missingDataStr
	}

	if !status.DropStats {
		dropsStats := dropsStats{
			DropsReceivedCurrent:              missingDataNum,
			DropsReceivedQoQ:                  missingDataNum,
			UniqueViewersReceivedADropCurrent: missingDataNum,
			UniqueViewersReceivedADropQoQ:     missingDataNum,
			ActiveCampaignsCurrent:            missingDataNum,
			ActiveCampaignsQoQ:                missingDataNum,
		}
		err := utils.Copy(&copiedData, &dropsStats)
		if err != nil {
			return copiedData, err
		}
	}

	if !status.AudienceStats {
		audienceStats := audienceStats{
			RisingCountriesByStreamers: missingDataStr,
			RisingCountriesByViewers:   missingDataStr,
			TopThreeGamesByStreamers:   missingDataStr,
			TopThreeGamesByViewers:     missingDataStr,
		}
		err := utils.Copy(&copiedData, &audienceStats)
		if err != nil {
			return copiedData, err
		}
	}

	return copiedData, nil
}

func addGameMetadata(resp *models.GameOnePagerDataResponse) {
	resp.GameInfo = models.GameInfo{
		GameID:   mainGameID,
		GameName: mainGameName,
	}
}

func makeTestOnePagerResponseWithMissingStatus(status models.OnePagerSectionPresentStatus, ifFillinDefault, ifFillGameMetadata bool) (models.GameOnePagerDataResponse, error) {
	copiedData := models.GameOnePagerDataResponse{}
	if ifFillGameMetadata {
		addGameMetadata(&copiedData)
	}

	copiedData.NumOfExtensions = onePagerResponseWithFullData.NumOfExtensions

	if !status.OverviewStats {
		copiedData.OverviewStats = nil
		if ifFillinDefault {
			copiedData.GeneralStats = &generalStats
		}
	} else {
		copiedData.OverviewStats = &models.OverviewStats{
			HoursWatched:                onePagerWithAllData.HoursWatchedCurrent,
			HoursWatchedChanges:         onePagerWithAllData.HoursWatchedQoQ,
			UniqueViewers:               onePagerWithAllData.UniqueViewersCurrent,
			UniqueViewersChanges:        onePagerWithAllData.UniqueViewersQoQ,
			ViewingSessions:             onePagerWithAllData.ViewingSessionsCurrent,
			ViewingSessionsChanges:      onePagerWithAllData.ViewingSessionsQoQ,
			HoursBroadcast:              onePagerWithAllData.HoursBroadcastCurrent,
			HoursBroadcastChanges:       onePagerWithAllData.HoursBroadcastQoQ,
			UniqueBroadcasters:          onePagerWithAllData.UniqueBroadcastersCurrent,
			UniqueBroadcastersChanges:   onePagerWithAllData.UniqueBroadcastersQoQ,
			BroadcastingSessions:        onePagerWithAllData.BroadcastingSessionsCurrent,
			BroadcastingSessionsChanges: onePagerWithAllData.BroadcastingSessionsQoQ,
		}
	}

	if !status.BroadcasterStats {
		copiedData.BroadcasterStats = nil
		if ifFillinDefault {
			copiedData.PopularBroadcasters = &[]models.PopularBroadcaster{
				{Name: "anbornsky", DisplayName: "AnbornSky"},
				{Name: "xocliw", DisplayName: "Xocliw"},
				{Name: "theplastereddragon", DisplayName: "theplastereddragon"},
			}
		}
	} else {
		copiedData.BroadcasterStats = &models.BroadcasterStats{
			TopBroadcaster: &models.TopBroadcaster{
				Name:              onePagerWithAllData.TopChannelLogin,
				DisplayName:       onePagerWithAllData.TopChannelDisplayName,
				HoursBroadcast:    onePagerWithAllData.TopChannelHoursBroadcast,
				HoursWatched:      onePagerWithAllData.TopChannelHoursWatched,
				ConcurrentViewers: onePagerWithAllData.TopChannelAvgCCU,
			},
			DedicatedBroadcaster: &models.DedicatedBroadcaster{
				Name:                       onePagerWithAllData.MostDedicatedChannelLogin,
				DisplayName:                onePagerWithAllData.MostDedicatedChannelDisplayname,
				StreamedSessions:           onePagerWithAllData.MostDedicatedChannelSessions,
				AvgStreamSessionsPerWeek:   0.33,
				PercentageOfHoursBroadcast: onePagerWithAllData.MostDedicatedChannelGamePercentage,
			},
			RisingStar: &models.RisingStar{
				Name:                 onePagerWithAllData.RisingChannelLogin,
				DisplayName:          onePagerWithAllData.RisingChannelDisplayname,
				HoursStreamed:        onePagerWithAllData.RisingChannelHoursBroadcast,
				HoursWatched:         onePagerWithAllData.RisingChannelHoursWatched,
				AvgConcurrentViewers: onePagerWithAllData.RisingChannelAvgCCU,
			},
		}
	}

	if !status.SocialStats {
		copiedData.SocialStats = nil
	} else {
		copiedData.SocialStats = &models.SocialStats{
			NumOfChatMessages:     onePagerWithAllData.TotalChatMessages,
			PercentOfEngagement:   onePagerWithAllData.ChatterPercent,
			NumOfPartnersStreamed: onePagerWithAllData.UniquePartnerAffiliate,
		}
	}

	if !status.ExtensionStats {
		copiedData.ExtensionStats = nil
		if ifFillinDefault {
			copiedData.PopularExtensions = &[]models.ExtensionStats{
				{Id: "2b9s9g75awk0ju391l0o2w4e7jxo47", Name: "popularExtension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")},
				{Id: "gh70y1spw727ohtgzbhc0hppvq9br2", Name: "popularExtension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")},
				{Id: "wi08ebtatdc7oj83wtl9uxwz807l8b", Name: "popularExtension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")},
			}
		}
	} else {
		copiedData.ExtensionStats = &[]models.ExtensionStats{
			{Id: extension1ID, Name: "extension1Name", Description: "extension1Description", Url: "extension1URL", IconURL: aws.String("extension1IconURL"), SplashURL: aws.String("extension1SplashURL")},
			{Id: extension2ID, Name: "extension2Name", Description: "extension2Description", Url: "extension2URL", IconURL: aws.String("extension2IconURL"), SplashURL: aws.String("extension2SplashURL")},
			{Id: extension3ID, Name: "extension3Name", Description: "extension3Description", Url: "extension3URL", IconURL: aws.String("extension3IconURL"), SplashURL: aws.String("extension3SplashURL")},
		}
	}

	if !status.DropStats {
		copiedData.DropsStats = nil
	} else {
		copiedData.DropsStats = &models.DropStats{
			NumOfDropsReceived:               onePagerWithAllData.DropsReceivedCurrent,
			NumOfDropsReceivedChange:         onePagerWithAllData.DropsReceivedQoQ,
			NumOfUniqueReceivingUsers:        onePagerWithAllData.UniqueViewersReceivedADropCurrent,
			NumOfUniqueReceivingUsersChanges: onePagerWithAllData.UniqueViewersReceivedADropQoQ,
			NumOfCampaigns:                   onePagerWithAllData.ActiveCampaignsCurrent,
			NumOfCampaignsChanges:            onePagerWithAllData.ActiveCampaignsQoQ,
		}
	}

	if !status.AudienceStats {
		copiedData.AudienceStats = nil
		if ifFillinDefault {
			copiedData.PopularGames = &models.PopularGames{
				PopularGamesToStream: []models.TwitchPopularGame{
					{Id: "1458", Name: "popularGameName1", Img: "popularGameImg1"},
					{Id: "4609", Name: "popularGameName2", Img: "popularGameImg2"},
					{Id: "3734", Name: "popularGameName3", Img: "popularGameImg3"},
				},
				PopularGamesToWatch: []models.TwitchPopularGame{
					{Id: "46091", Name: "popularGameName4", Img: "popularGameImg4"},
					{Id: "582", Name: "popularGameName5", Img: "popularGameImg5"},
					{Id: "37341", Name: "popularGameName6", Img: "popularGameImg6"},
				},
			}
		}
	} else {
		copiedData.AudienceStats = &models.AudienceStats{
			TopGamesToWatch: []models.TopGame{
				{Id: "21779", Name: "gameName1", Percentage: 9.531959895581132, Img: "gameImg1"},
				{Id: "138585", Name: "gameName2", Percentage: 5.079686475527248, Img: "gameImg2"},
				{Id: "488552", Name: "gameName3", Percentage: 4.368633406825281, Img: "gameImg3"},
			},
			TopGamesToStream: []models.TopGame{
				{Id: "32399", Name: "gameName4", Percentage: 14.848803747515351, Img: "gameImg4"},
				{Id: "4885529", Name: "gameNam5", Percentage: 9.942663558381568, Img: "gameImg5"},
				{Id: "217799", Name: "gameName6", Percentage: 9.67853060959945, Img: "gameImg6"},
			},
			TopBroadcasterCountries: []models.TopCountry{
				{Name: "United States", Percentage: 9.020, Code: "US"},
				{Name: "Japan", Percentage: 5.66666666667, Code: "JP"},
				{Name: "Afghanistan", Percentage: 1.69230769231, Code: "AF"},
			},
			TopViewerCountries: []models.TopCountry{
				{Name: "United States", Percentage: 3.50, Code: "US"},
				{Name: "Algeria", Percentage: 3.12564302379, Code: "DZ"},
				{Name: "Aland Islands", Percentage: 2.94245969016, Code: "AX"},
			},
		}
	}

	return copiedData, nil
}

func makeCacheDataString(resp models.GameOnePagerDataResponse, presence models.OnePagerSectionPresentStatus, allPresent bool) (string, error) {
	copiedData := models.GameOnePagerDataResponse{
		GameInfo:        resp.GameInfo,
		NumOfExtensions: resp.NumOfExtensions,
	}

	// void cache data if the presence state is a missing
	if presence.OverviewStats {
		copiedData.OverviewStats = resp.OverviewStats
	}

	if presence.AudienceStats {
		copiedData.AudienceStats = resp.AudienceStats
	}

	if presence.BroadcasterStats {
		copiedData.BroadcasterStats = resp.BroadcasterStats
	}

	if presence.DropStats {
		copiedData.DropsStats = resp.DropsStats
	}

	if presence.ExtensionStats {
		copiedData.ExtensionStats = resp.ExtensionStats
	}

	if presence.SocialStats {
		copiedData.SocialStats = resp.SocialStats
	}

	cachedData := models.OnePagerCacheData{
		PresentStatus: presence,
		DataResp:      copiedData,
		AllPresent:    allPresent,
	}

	cachedDataStr, err := json.Marshal(cachedData)
	return string(cachedDataStr), err
}
