package one_pager

import (
	"fmt"
	"strconv"
	"strings"
	"sync"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

// Transfer what we have in db storing in onePagerData into GameOnePagerDataResponse type
func (o onePagerImpl) transferToGameDataResponse(ctx context.Context, gameData models.OnePagerData, status *models.OnePagerSectionPresentStatus) (models.GameOnePagerDataResponse, error) {
	gameDataResponse := models.GameOnePagerDataResponse{}

	gameDataResponse.NumOfExtensions = gameData.TotalActiveExtensions

	if status.OverviewStats {
		overviewStats := &models.OverviewStats{
			HoursWatched:                gameData.HoursWatchedCurrent,
			HoursWatchedChanges:         gameData.HoursWatchedQoQ,
			UniqueViewers:               gameData.UniqueViewersCurrent,
			UniqueViewersChanges:        gameData.UniqueViewersQoQ,
			ViewingSessions:             gameData.ViewingSessionsCurrent,
			ViewingSessionsChanges:      gameData.ViewingSessionsQoQ,
			HoursBroadcast:              gameData.HoursBroadcastCurrent,
			HoursBroadcastChanges:       gameData.HoursBroadcastQoQ,
			UniqueBroadcasters:          gameData.UniqueBroadcastersCurrent,
			UniqueBroadcastersChanges:   gameData.UniqueBroadcastersQoQ,
			BroadcastingSessions:        gameData.BroadcastingSessionsCurrent,
			BroadcastingSessionsChanges: gameData.BroadcastingSessionsQoQ,
		}
		gameDataResponse.OverviewStats = overviewStats
	}

	if status.BroadcasterStats {
		broadcastersStats := &models.BroadcasterStats{}
		if gameData.TopChannelLogin != missingDataStr {
			topBroadcaster := &models.TopBroadcaster{
				Name:              gameData.TopChannelLogin,
				DisplayName:       gameData.TopChannelDisplayName,
				HoursBroadcast:    gameData.TopChannelHoursBroadcast,
				HoursWatched:      gameData.TopChannelHoursWatched,
				ConcurrentViewers: gameData.TopChannelAvgCCU,
			}
			broadcastersStats.TopBroadcaster = topBroadcaster
		}

		if gameData.MostDedicatedChannelLogin != missingDataStr {
			mostDedicatedBroadcaster := &models.DedicatedBroadcaster{
				Name:                       gameData.MostDedicatedChannelLogin,
				DisplayName:                gameData.MostDedicatedChannelDisplayname,
				StreamedSessions:           gameData.MostDedicatedChannelSessions,
				AvgStreamSessionsPerWeek:   float64(int(float64(gameData.MostDedicatedChannelSessions)/float64(12)*100)) / 100,
				PercentageOfHoursBroadcast: gameData.MostDedicatedChannelGamePercentage,
			}
			broadcastersStats.DedicatedBroadcaster = mostDedicatedBroadcaster
		}

		if gameData.RisingChannelLogin != missingDataStr {
			risingStarBroadcaster := &models.RisingStar{
				Name:                 gameData.RisingChannelLogin,
				DisplayName:          gameData.RisingChannelDisplayname,
				HoursStreamed:        gameData.RisingChannelHoursBroadcast,
				HoursWatched:         gameData.RisingChannelHoursWatched,
				AvgConcurrentViewers: gameData.RisingChannelAvgCCU,
			}
			broadcastersStats.RisingStar = risingStarBroadcaster
		}

		gameDataResponse.BroadcasterStats = broadcastersStats
	}

	if status.SocialStats {
		socialStats := &models.SocialStats{
			NumOfChatMessages:     gameData.TotalChatMessages,
			PercentOfEngagement:   gameData.ChatterPercent,
			NumOfPartnersStreamed: gameData.UniquePartnerAffiliate,
		}
		gameDataResponse.SocialStats = socialStats
	}

	if status.ExtensionStats {
		extensions := o.parseExtensions(ctx, gameData.MostPopularExtensions, status)
		if len(extensions) != 0 {
			gameDataResponse.ExtensionStats = &extensions
		} else {
			status.ExtensionStats = false
		}
	}

	if status.DropStats {
		dropsStats := &models.DropStats{
			NumOfDropsReceived:               gameData.DropsReceivedCurrent,
			NumOfDropsReceivedChange:         gameData.DropsReceivedQoQ,
			NumOfUniqueReceivingUsers:        gameData.UniqueViewersReceivedADropCurrent,
			NumOfUniqueReceivingUsersChanges: gameData.UniqueViewersReceivedADropQoQ,
			NumOfCampaigns:                   gameData.ActiveCampaignsCurrent,
			NumOfCampaignsChanges:            gameData.ActiveCampaignsQoQ,
		}

		gameDataResponse.DropsStats = dropsStats
	}

	if status.AudienceStats {
		gameDataResponse.AudienceStats = &models.AudienceStats{}
		gameDataResponse.AudienceStats.TopGamesToWatch = o.parseGame(ctx, gameData.TopThreeGamesByViewers)
		gameDataResponse.AudienceStats.TopGamesToStream = o.parseGame(ctx, gameData.TopThreeGamesByStreamers)
		gameDataResponse.AudienceStats.TopBroadcasterCountries = o.parseCountry(ctx, gameData.RisingCountriesByStreamers)
		gameDataResponse.AudienceStats.TopViewerCountries = o.parseCountry(ctx, gameData.RisingCountriesByViewers)
		if len(gameDataResponse.AudienceStats.TopGamesToWatch) == 0 && len(gameDataResponse.AudienceStats.TopGamesToStream) == 0 {
			status.AudienceStats = false
			gameDataResponse.AudienceStats = nil
		}
	}

	return gameDataResponse, nil
}

func (o onePagerImpl) parseExtensions(ctx context.Context, topExtensionsStr string, status *models.OnePagerSectionPresentStatus) []models.ExtensionStats {
	var extensionsStats [3]models.ExtensionStats
	extensionIds := strings.Split(topExtensionsStr, itemDelimiter)
	wg := sync.WaitGroup{}
	for i, extensionId := range extensionIds {
		wg.Add(1)
		go func(index int, id string) {
			defer handlePanicAndWaitGroup(ctx, &wg)
			if id == "" {
				return
			}
			popularExtension, err := o.emsClient.GetReleasedExtensionByID(ctx, id)
			if err != nil {
				if err != models.ErrResourceNotFound {
					logx.Error(ctx, fmt.Sprintf("error loading released extension for %s: %v", id, err))
				}
				return
			}
			extensionStats := models.ExtensionStats{
				Id:          id,
				Name:        popularExtension.Name,
				Description: popularExtension.Description,
				Url:         popularExtension.Url,
				IconURL:     popularExtension.IconURL,
				SplashURL:   popularExtension.SplashURL,
			}
			extensionsStats[index] = extensionStats
		}(i, extensionId)
	}

	wg.Wait()

	var processedExtensionsStats []models.ExtensionStats
	for _, extensionStats := range extensionsStats {
		if extensionStats.Id != "" {
			processedExtensionsStats = append(processedExtensionsStats, extensionStats)
		}
	}

	if len(processedExtensionsStats) == 0 && status != nil {
		status.ExtensionStats = false
	}
	return processedExtensionsStats
}

func (o onePagerImpl) parseGame(ctx context.Context, topGamesStr string) []models.TopGame {
	games := strings.Split(topGamesStr, itemDelimiter)
	wg := sync.WaitGroup{}
	if len(games) == 0 || games[0] == missingDataStr {
		return []models.TopGame{}
	} else {
		var topGames [3]models.TopGame
		for i, g := range games {
			wg.Add(1)
			go func(index int, game string) {
				defer handlePanicAndWaitGroup(ctx, &wg)
				gameAndPercentage := strings.Split(game, sectionDelimiter)
				if len(gameAndPercentage) != 2 {
					logx.Error(ctx, fmt.Sprintf("Error encoutered when trying to parse gameStr %s", topGamesStr))
					return
				}

				gameID := gameAndPercentage[0]
				if gameID == "" {
					return
				}

				percentage, err := strconv.ParseFloat(gameAndPercentage[1], 64)
				if err != nil {
					logx.Error(ctx, fmt.Sprintf("Error encoutered when trying to parse percentage %s for game %s: %v", gameAndPercentage[1], gameID, err))
					return
				}

				topGame, err := o.discoveryClient.GetGameByID(ctx, gameID)
				if err != nil {
					logx.Error(ctx, fmt.Sprintf("Error encoutered when trying to get game name for game %s: %v", gameID, err))
					return
				}

				topGame.Percentage = percentage
				topGames[index] = topGame
			}(i, g)
		}

		wg.Wait()

		var processedTopGames []models.TopGame
		for _, topGame := range topGames {
			if topGame.Id != "" {
				processedTopGames = append(processedTopGames, topGame)
			}
		}
		if len(processedTopGames) == 0 {
			return []models.TopGame{}
		}
		return processedTopGames
	}
}

func (o onePagerImpl) parseCountry(ctx context.Context, topCountriesStr string) []models.TopCountry {
	countries := strings.Split(topCountriesStr, itemDelimiter)
	if len(countries) == 0 || countries[0] == missingDataStr {
		return []models.TopCountry{}
	} else {
		var topCountries []models.TopCountry
		for _, game := range countries {
			countryAndPercentage := strings.Split(game, sectionDelimiter)
			if len(countryAndPercentage) != 3 {
				continue
			}

			countryName := countryAndPercentage[1]
			countryCode := countryAndPercentage[0]

			percentage, err := strconv.ParseFloat(countryAndPercentage[2], 64)
			if err != nil {
				logx.Error(ctx, fmt.Sprintf("Error encoutered when trying to parse percentage %s for country %s: %v", countryAndPercentage[1], countryName, err))
				continue
			}

			topCountry := models.TopCountry{
				Name:       countryName,
				Percentage: percentage,
				Code:       countryCode,
			}

			topCountries = append(topCountries, topCountry)
		}

		if len(topCountries) == 0 {
			return []models.TopCountry{}
		} else {
			return topCountries
		}
	}
}
