package one_pager

import (
	"fmt"
	"testing"
	"time"

	"code.justin.tv/insights/piper-service/models"

	cachemock "code.justin.tv/insights/piper-service/backend/mocks"
	s3reportmock "code.justin.tv/insights/piper-service/backend/s3report/mocks"
	discoverymock "code.justin.tv/insights/piper-service/internal/clients/discovery/mocks"
	emsmock "code.justin.tv/insights/piper-service/internal/clients/ems/mocks"
	piperdbmock "code.justin.tv/insights/piper-service/internal/clients/piperdb/mocks"
	rbacmock "code.justin.tv/insights/piper-service/internal/clients/rbac/mocks"
	s3mock "code.justin.tv/insights/piper-service/internal/clients/s3/mocks"
	spademock "code.justin.tv/insights/piper-service/internal/clients/spade/mocks"
	usersmock "code.justin.tv/insights/piper-service/internal/clients/users/mocks"
	piperconfig "code.justin.tv/insights/piper-service/internal/config"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

type onePagerSuite struct {
	suite.Suite

	piperDBClient   *piperdbmock.Client
	discoveryClient *discoverymock.Client
	spadeClient     *spademock.Client
	rbacClient      *rbacmock.Client
	usersClient     *usersmock.Client
	emsClient       *emsmock.Client
	s3Client        *s3mock.Client
	s3report        *s3reportmock.Backend
	cacher          *cachemock.Cacher

	onePagerImpl *onePagerImpl

	userID      string
	gameID      string
	companyIDA  string
	companyIDB  string
	companyName string

	startDate  time.Time
	yearStr    string
	quarterStr string

	dateCacheKey         string
	onePagerDataCacheKey string

	noGameUser string
	config     piperconfig.PiperConfig

	ctx context.Context
}

func (suite *onePagerSuite) SetupTest() {
	suite.piperDBClient = &piperdbmock.Client{}
	suite.discoveryClient = &discoverymock.Client{}
	suite.spadeClient = &spademock.Client{}
	suite.rbacClient = &rbacmock.Client{}
	suite.usersClient = &usersmock.Client{}
	suite.emsClient = &emsmock.Client{}
	suite.s3Client = &s3mock.Client{}
	suite.s3report = &s3reportmock.Backend{}
	suite.cacher = &cachemock.Cacher{}

	suite.onePagerImpl = &onePagerImpl{
		suite.piperDBClient,
		suite.discoveryClient,
		suite.spadeClient,
		suite.rbacClient,
		suite.usersClient,
		suite.emsClient,
		suite.s3Client,
		suite.s3report,
		suite.cacher,
	}

	suite.userID = "131180911"
	suite.gameID = mainGameID
	suite.companyIDA = "9620a89d-b07c-4160-a851-ae832fa6985e"
	suite.companyIDB = "c81b5de6-20e1-4745-9f02-795637eb3e89"
	suite.companyName = "Blizzard"

	suite.yearStr = "2018"
	suite.quarterStr = "1"
	suite.startDate = time.Date(2018, time.Month(1), 1, 0, 0, 0, 0, time.UTC)
	suite.dateCacheKey = yearQuarterKeyPrefix + suite.yearStr + cacheDelimiter + suite.quarterStr
	suite.onePagerDataCacheKey = fmt.Sprintf("%s%s_%s_%s", onePagerDataPrefix, suite.gameID, suite.yearStr, suite.quarterStr)

	suite.config = piperconfig.PiperConfig{ReportBucket: "", ReportPrefix: ""}
	suite.noGameUser = "270724749"
	suite.ctx = context.Background()
}

func (suite *onePagerSuite) TearDownTest() {
	suite.piperDBClient.AssertExpectations(suite.T())
	suite.discoveryClient.AssertExpectations(suite.T())
	suite.spadeClient.AssertExpectations(suite.T())
	suite.rbacClient.AssertExpectations(suite.T())
	suite.usersClient.AssertExpectations(suite.T())
	suite.emsClient.AssertExpectations(suite.T())
	suite.s3Client.AssertExpectations(suite.T())
	suite.s3report.AssertExpectations(suite.T())
	suite.cacher.AssertExpectations(suite.T())
}

type UserCanAccessTest struct {
	onePagerSuite
}

func (suite *UserCanAccessTest) TestNonWhitelistedDeveloperCanAccessOwnResource() {
	suite.rbacClient.On("Validate", suite.ctx, mock.Anything).Return(nil)

	isWhitelisted, err := suite.onePagerImpl.UserCanAccess(suite.ctx, suite.userID, suite.gameID, suite.companyIDA)
	suite.NoError(err)
	suite.False(isWhitelisted)
}

func (suite *UserCanAccessTest) TestWhitelistedDeveloperCanAccessAnyResource() {
	isWhitelisted, err := suite.onePagerImpl.UserCanAccess(suite.ctx, "131180913", suite.gameID, "")
	suite.NoError(err)
	suite.True(isWhitelisted)
}

func (suite *UserCanAccessTest) TestNonWhitelistedUserCantAccessOtherCompanyResource() {
	suite.rbacClient.On("Validate", suite.ctx, mock.Anything).Return(models.ErrAccessForbidden)

	isWhitelisted, err := suite.onePagerImpl.UserCanAccess(suite.ctx, suite.userID, suite.gameID, suite.companyIDA)
	suite.Error(err)
	suite.False(isWhitelisted)
}

func (suite *UserCanAccessTest) TestNoGameUserCantAccessResource() {
	suite.rbacClient.On("Validate", suite.ctx, mock.Anything).Return(models.ErrAccessForbidden)

	isWhitelisted, err := suite.onePagerImpl.UserCanAccess(suite.ctx, suite.noGameUser, suite.gameID, suite.companyIDA)
	suite.Error(err)
	suite.False(isWhitelisted)
}

func TestUserCanAccessSuite(t *testing.T) {
	suite.Run(t, new(UserCanAccessTest))
}
