package one_pager

import (
	"strconv"
	"sync"

	"time"

	"fmt"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

const (
	itemDelimiter        = "||"
	sectionDelimiter     = ","
	shortDateFormat      = "2006-01-02"
	yearQuarterKeyPrefix = "yq_"
)

func checkIfMissingFields(gameData models.OnePagerData) (models.OnePagerSectionPresentStatus, bool) {
	status := models.OnePagerSectionPresentStatus{
		OverviewStats:    false,
		BroadcasterStats: false,
		SocialStats:      false,
		ExtensionStats:   false,
		DropStats:        false,
		AudienceStats:    false,
	}

	// present status for overview stats section
	if gameData.HoursBroadcastCurrent != missingDataNum || gameData.HoursWatchedCurrent != missingDataNum || gameData.UniqueBroadcastersCurrent != missingDataNum || gameData.UniqueViewersCurrent != missingDataNum || gameData.BroadcastingSessionsCurrent != missingDataNum || gameData.ViewingSessionsCurrent != missingDataNum {
		status.OverviewStats = true
	}

	// present status for broadcaster stats section
	if !ifStrFieldMissing(gameData.TopChannelLogin) || !ifStrFieldMissing(gameData.MostDedicatedChannelLogin) || !ifStrFieldMissing(gameData.RisingChannelLogin) {
		status.BroadcasterStats = true
	}

	// present status for social impact section
	if gameData.TotalChatMessages != missingDataNum || gameData.UniqueChatters != missingDataNum || gameData.UniquePartnerAffiliate != missingDataNum {
		status.SocialStats = true
	}

	// present status for extension section
	if !ifStrFieldMissing(gameData.MostPopularExtensions) {
		status.ExtensionStats = true
	}

	// present status for drops section
	if gameData.DropsReceivedCurrent != missingDataNum || gameData.UniqueViewersReceivedADropCurrent != missingDataNum || gameData.ActiveCampaignsCurrent != missingDataNum {
		status.DropStats = true
	}

	// present status for audience insights
	if !ifStrFieldMissing(gameData.TopThreeGamesByStreamers) || !ifStrFieldMissing(gameData.TopThreeGamesByViewers) {
		status.AudienceStats = true
	}

	// If all fields are present
	if status.OverviewStats && status.BroadcasterStats && status.SocialStats && status.ExtensionStats && status.DropStats && status.AudienceStats {
		return status, true
	}

	return status, false
}

func ifStrFieldMissing(s string) bool {
	if s == missingDataStr || s == "" {
		println(s)
		return true
	}
	return false
}

func convertTimeToStr(t time.Time) string {
	return t.Format(shortDateFormat)
}

func ifStatsMissing(statsStatus bool) bool {
	if !statsStatus {
		return true
	}
	return false
}

func ifValueIsNil(value interface{}) bool {
	if value != nil {
		return false
	}
	return true
}

func wrapResults(key string, val interface{}) map[string]interface{} {
	wrapped := make(map[string]interface{})
	wrapped[key] = val
	return wrapped
}

func convertFloatToStr(field float64) string {
	return strconv.FormatFloat(field, 'f', 2, 64)
}

func convertIntToStr(field uint64) string {
	return strconv.FormatInt(int64(field), 10)
}

func appendHeader(header *[]string, column string) {
	*header = append(*header, column)
}

func appendContent(content *[]string, data string) {
	*content = append(*content, data)
}

func (o onePagerImpl) ConvertYearQuarterToStartDate(ctx context.Context, yearStr, quarterStr string) (time.Time, error) {
	key := yearQuarterKeyPrefix + yearStr + cacheDelimiter + quarterStr
	startStr, found := o.cacher.GetStringProperties(ctx, key)
	if found {
		starTime, err := time.Parse(shortDateFormat, startStr)
		if err != nil {
			return time.Time{}, err
		}
		return starTime, nil
	}

	startTime, err := o.piperDBClient.GetQuarterStartByYearQuarter(ctx, yearStr, quarterStr)
	if err != nil {
		return startTime, err
	}

	err = o.cacher.CacheStringProperties(ctx, key, convertTimeToStr(startTime))
	if err != nil {
		logx.Error(ctx, fmt.Sprintf("failed to cache quarter start time for %s in cache", key))
	}
	return startTime, nil
}

func handlePanicAndWaitGroup(ctx context.Context, wg *sync.WaitGroup) {
	wg.Done()
	if r := recover(); r != nil {
		message := fmt.Sprintf("panic in async job: %s", r)
		logx.Error(ctx, message)
	}
}
