package one_pager

import (
	"testing"

	cachemock "code.justin.tv/insights/piper-service/backend/mocks"
	s3reportmock "code.justin.tv/insights/piper-service/backend/s3report/mocks"
	discoverymock "code.justin.tv/insights/piper-service/internal/clients/discovery/mocks"
	emsmock "code.justin.tv/insights/piper-service/internal/clients/ems/mocks"
	piperdbmock "code.justin.tv/insights/piper-service/internal/clients/piperdb/mocks"
	rbacmock "code.justin.tv/insights/piper-service/internal/clients/rbac/mocks"
	s3mock "code.justin.tv/insights/piper-service/internal/clients/s3/mocks"
	spademock "code.justin.tv/insights/piper-service/internal/clients/spade/mocks"
	usersmock "code.justin.tv/insights/piper-service/internal/clients/users/mocks"
	"code.justin.tv/insights/piper-service/internal/utils"
	"code.justin.tv/insights/piper-service/models"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

type OnePagerUtilsTest struct {
	suite.Suite

	piperDBClient   *piperdbmock.Client
	discoveryClient *discoverymock.Client
	spadeClient     *spademock.Client
	rbacClient      *rbacmock.Client
	usersClient     *usersmock.Client
	emsClient       *emsmock.Client
	s3Client        *s3mock.Client
	s3report        *s3reportmock.Backend
	cacher          *cachemock.Cacher

	onePagerImpl *onePagerImpl
	ctx          context.Context
}

func (suite *OnePagerUtilsTest) SetupTest() {
	var err error

	suite.piperDBClient = &piperdbmock.Client{}
	suite.discoveryClient = &discoverymock.Client{}
	suite.spadeClient = &spademock.Client{}
	suite.rbacClient = &rbacmock.Client{}
	suite.usersClient = &usersmock.Client{}
	suite.emsClient = &emsmock.Client{}
	suite.s3Client = &s3mock.Client{}
	suite.s3report = &s3reportmock.Backend{}
	suite.cacher = &cachemock.Cacher{}

	suite.onePagerImpl = &onePagerImpl{
		suite.piperDBClient,
		suite.discoveryClient,
		suite.spadeClient,
		suite.rbacClient,
		suite.usersClient,
		suite.emsClient,
		suite.s3Client,
		suite.s3report,
		suite.cacher,
	}
	suite.NoError(err)

	suite.ctx = context.Background()
}

func (suite *OnePagerUtilsTest) TestCheckingMissingAllData() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    false,
		BroadcasterStats: false,
		SocialStats:      false,
		ExtensionStats:   false,
		DropStats:        false,
		AudienceStats:    false,
	}

	missingAllDataResp, err := makeTestOnePagerDataBasedOnMissingStatus(presence)
	suite.NoError(err)

	status, ifAllPresent := checkIfMissingFields(missingAllDataResp)
	suite.False(ifAllPresent)

	if diff := utils.Equal(status, presence); diff != nil {
		suite.T().Error(diff)
	}
}

func (suite *OnePagerUtilsTest) TestCheckingMissingNoData() {
	presence := models.OnePagerSectionPresentStatus{
		OverviewStats:    true,
		BroadcasterStats: true,
		SocialStats:      true,
		ExtensionStats:   true,
		DropStats:        true,
		AudienceStats:    true,
	}

	missingAllDataResp, err := makeTestOnePagerDataBasedOnMissingStatus(presence)
	suite.NoError(err)

	status, ifAllPresent := checkIfMissingFields(missingAllDataResp)
	suite.True(ifAllPresent)

	if diff := utils.Equal(status, presence); diff != nil {
		suite.T().Error(diff)
	}
}

func TestOnePagerUtilsSuite(t *testing.T) {
	suite.Run(t, new(OnePagerUtilsTest))
}
