package s3report

import (
	"encoding/csv"
	"os"
	"strings"

	uuid "github.com/satori/go.uuid"
	"golang.org/x/net/context"
	"golang.org/x/text/encoding/charmap"
)

func (c *s3ReportBackendImpl) SaveReportToS3(ctx context.Context, report [][]string, domainKey, reportType, startDate, endDate, bucket, prefix string) (string, error) {
	localFilePath := getFileNameCSV(domainKey, reportType, startDate, endDate)

	err := saveReportToCSV(report, localFilePath)
	if err != nil {
		return "", err
	}

	filename, err := c.s3client.UploadFileToS3(bucket, prefix, localFilePath)
	if err != nil {
		return "", err
	}

	err = cleanUp(localFilePath)
	if err != nil {
		return "", err
	}

	return filename, nil
}

func getFileNameCSV(domainKey, reportType, startDate, endDate string) string {
	ext := ".csv"

	id := uuid.NewV4().String()
	elem := []string{toISO8859_1(domainKey), reportType, startDate, endDate, id}
	filename := strings.Join(elem, "_")
	filename = strings.Replace(filename, "/", "-", -1)
	return filename + ext
}

// S3 requires strings be ISO-8859-1. Since the extension name is submitted
// by users, it may not be compliant. Strip any non ISO-8859-1 characters.
func toISO8859_1(input string) string {
	sanitizedLetters := make([]string, len(input))
	for _, letter := range input {
		_, err := charmap.ISO8859_1.NewEncoder().String(string(letter))
		if err != nil {
			continue
		}

		// use original form of letter
		sanitizedLetters = append(sanitizedLetters, string(letter))
	}
	return strings.Join(sanitizedLetters, "")
}

func saveReportToCSV(report [][]string, fileName string) (err error) {
	file, err := os.Create(fileName)
	if err != nil {
		return err
	}

	defer func() {
		cerr := file.Close()
		if err == nil {
			err = cerr
		}
	}()

	writer := csv.NewWriter(file)
	defer writer.Flush()

	for _, row := range report {
		if err := writer.Write(row); err != nil {
			return err
		}
	}
	return nil
}

func cleanUp(fileName string) error {
	return os.Remove(fileName)
}
