// tslint:disable-next-line: no-submodule-imports
import 'source-map-support/register';

import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import { VpcStack } from './stacks/vpc';
import { AlbStack, AlbStackProps } from './stacks/alb';
import { PrivateLinkServiceStack } from './stacks/private-link-service';
import { TwitchDnsResolverStack } from './stacks/twitch-dns-resolver';

interface EnvProps {
  prefix: string;
  env: cdk.Environment;
  vpcId: string;
  vpcPrivSubnets: ec2.SubnetAttributes[];
  alb: AlbStackProps;
  privLinkWhitelistGql: string; // GraphQL ARN principal allowed to the private link service endpoint
  twitchDnsServers: string[]; // https://wiki.twitch.com/display/PS/What+DNS+servers+should+be+used
}

const app = new cdk.App();

// Dev
createStacks(app, {
  prefix: 'Dev',
  env: {
    account: '895328872748',
    region: 'us-west-2',
  },
  vpcId: 'vpc-2d57844b',
  vpcPrivSubnets: [
    { subnetId: 'subnet-bf46abd9', availabilityZone: 'us-west-2a', routeTableId: 'rtb-36072650' },
    { subnetId: 'subnet-1380875a', availabilityZone: 'us-west-2b', routeTableId: 'rtb-02052464' },
    { subnetId: 'subnet-518f0b0a', availabilityZone: 'us-west-2c', routeTableId: 'rtb-2907264f' },
  ],
  alb: {
    loadBalancerArn: 'arn:aws:elasticloadbalancing:us-west-2:895328872748:loadbalancer/app/awseb-AWSEB-M2UKI8N4PH6C/db6577a6907b9cb1',
    loadBalancerDnsName: 'internal-awseb-AWSEB-M2UKI8N4PH6C-1199737523.us-west-2.elb.amazonaws.com',
    securityGroupId: 'sg-ffcca281',
    listenerArn: 'arn:aws:elasticloadbalancing:us-west-2:895328872748:listener/app/awseb-AWSEB-M2UKI8N4PH6C/db6577a6907b9cb1/4cb775a6f6443e91',
  },
  privLinkWhitelistGql: "arn:aws:iam::645130450452:root", // twitch-graphql-dev
  twitchDnsServers: ['10.202.125.133', '10.202.124.213', '10.202.126.110', '10.202.124.10'],
});

// Prod
createStacks(app, {
  prefix: 'Prod',
  env: {
    account: '452744562037',
    region: 'us-west-2',
  },
  vpcId: 'vpc-dc5a89ba',
  vpcPrivSubnets: [
    { subnetId: 'subnet-a08d8ae9', availabilityZone: 'us-west-2a', routeTableId: 'rtb-e60a2b80' },
    { subnetId: 'subnet-da58b5bc', availabilityZone: 'us-west-2b', routeTableId: 'rtb-a20b2ac4' },
    { subnetId: 'subnet-2495117f', availabilityZone: 'us-west-2c', routeTableId: 'rtb-860f2ee0' },
  ],
  alb: {
    loadBalancerArn: 'arn:aws:elasticloadbalancing:us-west-2:452744562037:loadbalancer/app/awseb-AWSEB-WHIB16XMDGQ5/a8da34ff00be7816',
    loadBalancerDnsName: 'internal-awseb-AWSEB-WHIB16XMDGQ5-322674940.us-west-2.elb.amazonaws.com',
    securityGroupId: 'sg-74b1f40a',
    listenerArn: 'arn:aws:elasticloadbalancing:us-west-2:452744562037:listener/app/awseb-AWSEB-WHIB16XMDGQ5/a8da34ff00be7816/e0efa368e3ff9ffc',
  },
  privLinkWhitelistGql: "arn:aws:iam::787149559823:root", // twitch-graphql-prod
  twitchDnsServers: ['10.197.231.47', '10.197.232.103', '10.197.223.101', '10.197.218.250'],
});

// createStacks for each environment
function createStacks(scope: cdk.App, props: EnvProps) {
  const { prefix, env } = props;

  const vpcStack = new VpcStack(scope, prefix + 'Vpc', {
    env,
    vpcId: props.vpcId,
    vpcPrivSubnets: props.vpcPrivSubnets,
  });

  const albStack = new AlbStack(scope, prefix + 'Alb', {
    env,
    ...props.alb
  });

  new PrivateLinkServiceStack(scope, prefix + 'PrivateLinkServiceForGraphQL', {
    env,
    vpc: vpcStack.vpc,
    vpcPrivSubnets: vpcStack.vpcPrivSubnets,
    alb: albStack.alb,
    albListener: albStack.albListener,
    whitelistedArnPrincipal: props.privLinkWhitelistGql,
  })

  new TwitchDnsResolverStack(scope, prefix + 'TwitchDnsResolver', {
    env,
    twitchDnsServers: props.twitchDnsServers,
    vpc: vpcStack.vpc,
    vpcPrivSubnets: vpcStack.vpcPrivSubnets,
  });
}
