import * as ec2 from '@aws-cdk/aws-ec2';
import * as elbv2 from '@aws-cdk/aws-elasticloadbalancingv2';
import * as cdk from '@aws-cdk/core';
import * as iam from '@aws-cdk/aws-iam';
import * as easymode from '@twitch/easymode';


interface PrivateLinkServiceStackProps extends cdk.StackProps {
  vpc: ec2.IVpc;
  vpcPrivSubnets: ec2.ISubnet[];
  alb: elbv2.IApplicationLoadBalancer;
  albListener: elbv2.IApplicationListener;
  whitelistedArnPrincipal: string;
}

export class PrivateLinkServiceStack extends cdk.Stack {
  constructor(scope: cdk.Construct, name: string, props: PrivateLinkServiceStackProps) {
    super(scope, name, props);

    // NLB pointing to the ALB's static IP addresses.
    // https://aws.amazon.com/blogs/networking-and-content-delivery/using-static-ip-addresses-for-application-load-balancers/
    const endpointNlb = new easymode.EndpointServiceLoadBalancer(this, 'endpointLambda2', {
      alb: props.alb,
      albListener: props.albListener,
      albListenerPort: 443, // https
      protocol: elbv2.ApplicationProtocol.HTTPS,
      vpc: props.vpc,
      cidr: props.vpc.vpcCidrBlock,
      nlbProps: {
        internetFacing: false,
        vpcSubnets: { subnets: props.vpcPrivSubnets }
      },
    });

    // Private Link endpoint service
    new ec2.VpcEndpointService(this, "EndpointService", {
      vpcEndpointServiceLoadBalancers: [endpointNlb.nlb],
      acceptanceRequired: false,
      whitelistedPrincipals: [new iam.ArnPrincipal(props.whitelistedArnPrincipal)],
    });
  }
}
